<?php

namespace app\subcomponents\students\UseCases\ReEnrolment;

use common\models\AcademicOffering;
use common\models\AcademicOfferingModel;
use common\models\ApplicationModel;
use common\models\Application;
use common\models\ApplicationCapesubject;
use common\models\ApplicationPeriod;
use common\models\CapeGroup as DBCapeGroup;
use common\models\Offer;
use common\models\StudentRegistration;
use common\models\UserModel;
use common\models\ProgrammeSpecialisation;
use common\models\AcademicOfferingSpecialisation;
use yii\helpers\Url;

class ReEnrolmentViewModel
{
    public $pageTitle;
    public $breadcrumbs;
    public $registrationSummary;
    public $applicationPeriods;
    public $applicationPeriodButtonLabel;
    public $academicOfferings;
    public $programmeButtonLabel;
    public $academicOfferingIdSelected;
    public $applicationCapesubjectForms;
    public $capeGroups;
    public $displayCapeSelectionForm;
    public $displaySpecialisationForm;
    public $isGeneralSpecialisationMandatory;
    public $isMajorMandatory;
    public $isMinorMandatory;
    public $displayGeneralSpecialisations;
    public $displayMajors;
    public $displayMinors;
    public $generalSpecialisations;
    public $majors;
    public $minors;
    public $programmeSpecialisationChoiceForm;
    public $detailsForm;

    private function __construct(
        $pageTitle,
        $breadcrumbs,
        $registrationSummary,
        $applicationPeriods,
        $applicationPeriodButtonLabel,
        $academicOfferings,
        $programmeButtonLabel,
        $academicOfferingIdSelected,
        $applicationCapesubjectForms,
        $capeGroups,
        $displayCapeSelectionForm,
        $displaySpecialisationForm,
        $isGeneralSpecialisationMandatory,
        $isMajorMandatory,
        $isMinorMandatory,
        $displayGeneralSpecialisations,
        $displayMajors,
        $displayMinors,
        $generalSpecialisations,
        $majors,
        $minors,
        $programmeSpecialisationChoiceForm,
        $detailsForm
    ) {
        $this->pageTitle = $pageTitle;
        $this->breadcrumbs =  $breadcrumbs;
        $this->registrationSummary = $registrationSummary;
        $this->applicationPeriods = $applicationPeriods;
        $this->applicationPeriodButtonLabel = $applicationPeriodButtonLabel;
        $this->academicOfferings = $academicOfferings;
        $this->programmeButtonLabel = $programmeButtonLabel;
        $this->academicOfferingIdSelected = $academicOfferingIdSelected;
        $this->applicationCapesubjectForms = $applicationCapesubjectForms;
        $this->capeGroups = $capeGroups;
        $this->displayCapeSelectionForm = $displayCapeSelectionForm;
        $this->displaySpecialisationForm = $displaySpecialisationForm;
        $this->isGeneralSpecialisationMandatory = $isGeneralSpecialisationMandatory;
        $this->isMajorMandatory = $isMajorMandatory;
        $this->isMinorMandatory = $isMinorMandatory;
        $this->displayGeneralSpecialisations = $displayGeneralSpecialisations;
        $this->displayMajors = $displayMajors;
        $this->displayMinors = $displayMinors;
        $this->generalSpecialisations = $generalSpecialisations;
        $this->majors = $majors;
        $this->minors = $minors;
        $this->programmeSpecialisationChoiceForm = $programmeSpecialisationChoiceForm;
        $this->detailsForm = $detailsForm;
    }

    public static function create(
        $personId,
        $studentRegistrationId,
        $applicationPeriodId,
        $academicOfferingId
    ) {
        $pageTitle = "Add Enrolment";
        $user = UserModel::findUserByID($personId);
        $fullName = UserModel::getUserFullname($user);
        $username = $user->username;

        $studentRegistration =
            StudentRegistration::find()
            ->where([
                "studentregistrationid" => $studentRegistrationId,
                "isactive" => 1,
                "isdeleted" => 0
            ])
            ->one();
        $offer =
            Offer::find()
            ->where([
                "offerid" => $studentRegistration->offerid,
                "isactive" => 1,
                "isdeleted" => 0
            ])
            ->one();
        $application =
            Application::find()
            ->where([
                "applicationid" => $offer->applicationid,
                "isactive" => 1,
                "isdeleted" => 0
            ])
            ->one();
        $programmeDescription = ApplicationModel::getProgrammeName($application);

        $breadcrumbs =
            new Breadcrumbs($personId, $studentRegistrationId, $fullName);

        $registrationSummary =
            new RegistrationSummary($username, $fullName, $programmeDescription);

        $applicationPeriods = self::prepareApplicationPeriodsDropdownlist(
            $personId,
            $studentRegistrationId
        );

        $applicationPeriodButtonLabel =
            self::getApplicationPeriodButtonLabel($applicationPeriodId);

        $academicOfferings = array();
        if ($applicationPeriodId != null) {
            $academicOfferings = self::prepareAcademicOfferingsDropdownlist(
                $personId,
                $studentRegistrationId,
                $applicationPeriodId
            );
        }

        $applicationCapesubjectForms = array();
        $capeGroups = array();
        $programmeButtonLabel = "Select Programme";
        $academicOfferingIdSelected = false;
        $displayCapeSelectionForm = false;
        $displaySpecialisationForm = false;
        $isSpecialisationSelectionAvailable = null;
        $isGeneralSpecialisationMandatory = null;
        $isMajorMandatory = null;
        $isMinorMandatory = null;
        $generalSpecialisations = null;
        $majors = null;
        $minors = null;
        $displayGeneralSpecialisations = false;
        $displayMajors = false;
        $displayMinors = false;

        if ($academicOfferingId != null) {
            $programmeButtonLabel =
                self::getProgrammeButtonLabel($academicOfferingId);

            $academicOfferingIdSelected = true;
            $academicOffering =
                AcademicOffering::find()
                ->where([
                    "academicofferingid" => $academicOfferingId,
                    "isactive" => 1,
                    "isdeleted" => 0
                ])
                ->one();
            if ($academicOffering->programmecatalogid == 10) {
                $applicationCapesubjectForms = self::generateApplicationCapesubjects();
                $capeGroups = self::generateCapeGroups($academicOfferingId);
                $displayCapeSelectionForm = true;
            }

            //Specializations Logic
            $isSpecialisationSelectionAvailable = $academicOffering->is_specialisation_selection_available;
            $isGeneralSpecialisationMandatory = $academicOffering->is_general_specialisation_mandatory;
            $isMajorMandatory = $academicOffering->is_major_mandatory;
            $isMinorMandatory = $academicOffering->is_minor_mandatory;

            if (
                $isSpecialisationSelectionAvailable == true
                && self::academicOfferingHasSpecialisations($academicOfferingId) == true
            ) {
                $displaySpecialisationForm = true;
                $generalSpecialisations = self::getGeneralSpecialisations($academicOfferingId);
                $majors = self::getMajors($academicOfferingId);
                $minors = self::getMinors($academicOfferingId);
            }

            $displayGeneralSpecialisations = !empty($generalSpecialisations) ? true : false;
            $displayMajors = !empty($majors) ? true : false;
            $displayMinors = !empty($minors) ? true : false;
        }

        $programmeSpecialisationChoiceForm  = ProgrammeSpecialisationChoiceForm::create();
        $detailsForm = new DetailsForm();

        // $reEnrolmentForm = new ReEnrolmentForm();

        return new ReEnrolmentViewModel(
            $pageTitle,
            $breadcrumbs,
            $registrationSummary,
            $applicationPeriods,
            $applicationPeriodButtonLabel,
            $academicOfferings,
            $programmeButtonLabel,
            $academicOfferingIdSelected,
            $applicationCapesubjectForms,
            $capeGroups,
            $displayCapeSelectionForm,
            $displaySpecialisationForm,
            $isGeneralSpecialisationMandatory,
            $isMajorMandatory,
            $isMinorMandatory,
            $displayGeneralSpecialisations,
            $displayMajors,
            $displayMinors,
            $generalSpecialisations,
            $majors,
            $minors,
            $programmeSpecialisationChoiceForm,
            $detailsForm
            // $reEnrolmentForm
        );
    }


    public static function prepareApplicationPeriodsDropdownlist(
        $userId,
        $studentRegistrationId
    ) {
        $options = array();

        $periods =
            ApplicationPeriod::find()
            ->where([
                "applicationperiodstatusid" => [5, 6],
                "isactive" => 1, "isdeleted" => 0
            ])
            ->all();

        if (!empty($periods)) {
            foreach ($periods as $period) {
                $href =
                    Url::to([
                        "re-enrolment/execute",
                        "personId" => $userId,
                        "studentRegistrationId" => $studentRegistrationId,
                        "applicationPeriodId" => $period->applicationperiodid
                    ]);
                $options[$period->name] = $href;
            }
        }
        return $options;
    }

    public function minimumElementHeight($numProgrammes = 0)
    {
        $periodDropdownButtonHeight = 35;
        $periodElementHeight = 30;
        $periodTotalHeight =
            $periodDropdownButtonHeight
            + (count($this->applicationPeriods) * $periodElementHeight);

        $programmeTotalHeight = 0;
        $commentsFieldHeight = 0;
        if ($numProgrammes != 0) {
            $programmeDropdownButtonHeight = 35;
            $programmeElementHeight = 30;
            $programmeTotalHeight =
                $programmeDropdownButtonHeight
                + ($numProgrammes * $programmeElementHeight);

            $commentsFieldHeight = 300;
        }

        $totalHeight =
            $periodTotalHeight + $programmeTotalHeight + $commentsFieldHeight;

        return "min-height:{$totalHeight}px";
    }

    public static function prepareAcademicOfferingsDropdownlist(
        $userId,
        $studentRegistrationId,
        $applicationPeriodId
    ) {
        $options = array();
        $academicOfferings =
            AcademicOffering::find()
            ->where([
                "applicationperiodid" => $applicationPeriodId,
                "isactive" => 1,
                "isdeleted" => 0
            ])
            ->all();
        if (!empty($academicOfferings)) {
            foreach ($academicOfferings as $academicOffering) {
                $href =
                    Url::to([
                        "re-enrolment/execute",
                        "personId" => $userId,
                        "studentRegistrationId" => $studentRegistrationId,
                        "applicationPeriodId" => $applicationPeriodId,
                        "academicOfferingId" => $academicOffering->academicofferingid
                    ]);

                $programmeName = AcademicOfferingModel::getProgrammeName($academicOffering);

                $options[$programmeName] = $href;
            }
        }

        return $options;
    }

    public static function getApplicationPeriodButtonLabel($applicationPeriodId)
    {
        if ($applicationPeriodId == null) {
            return "Select Application Period";
        } else {
            return ApplicationPeriod::find()
                ->where(["applicationperiodid" => $applicationPeriodId])
                ->one()
                ->name;
        }
    }

    public static function getProgrammeButtonLabel($academicOfferingId)
    {
        if ($academicOfferingId == null) {
            return "Select Programme";
        } else {
            $academicOffering =
                AcademicOffering::find()
                ->where([
                    "academicofferingid" => $academicOfferingId,
                    "isactive" => 1,
                    "isdeleted" => 0
                ])
                ->one();
            return AcademicOfferingModel::getProgrammeName($academicOffering);
        }
    }

    private static function generateApplicationCapesubjects()
    {
        $applicationCapesubjectForms = array();

        $capeGroups =
            DBCapeGroup::find()
            ->where(['isactive' => 1, 'isdeleted' => 0])
            ->all();

        if (empty($capeGroups)) {
            return $applicationCapesubjectForms;
        }

        foreach ($capeGroups as $capeGroup) {
            $record = new ApplicationCapesubject();
            //Values giving default value so as to facilitate validation (selective saving will be implemented)
            $record->capesubjectid = 0;
            $record->applicationid = 0;
            $applicationCapesubjectForms[] = $record;
        }
        return $applicationCapesubjectForms;
    }

    private static function generateCapeGroups($academicOfferingId)
    {
        $groups = array();

        $capeGroups =
            DBCapeGroup::find()
            ->where(["isactive" => 1, "isdeleted" => 0])
            ->all();

        if (empty($capeGroups)) {
            return $groups;
        }

        foreach ($capeGroups as $capeGroup) {
            $groups[] = CapeGroup::create($capeGroup, $academicOfferingId);
        }
        return $groups;
    }

    private static function academicOfferingHasSpecialisations($academicOfferingId)
    {
        return AcademicOfferingSpecialisation::find()
            ->where([
                "academic_offering_id" => $academicOfferingId,
                "is_deleted" => 0
            ])
            ->all();
    }

    /**
     * Returns associative array of programme specialisations offered in a
     * particular academic offering.
     * ["AcademicOfferingSpecialisation->id" => "ProgrammeSpecification->name']
     *
     * @param integer $academicOfferingId
     * @return array
     */
    private static function getGeneralSpecialisations($academicOfferingId)
    {
        $specialisations = array();

        $academicOfferingSpecialisations =
            AcademicOfferingSpecialisation::find()
            ->innerJoin(
                "programme_specialisation_type",
                "`academic_offering_specialisation`.`programme_specialisation_type_id` = `programme_specialisation_type`.`id`"
            )
            ->where(
                [
                    "academic_offering_specialisation.academic_offering_id" => $academicOfferingId,
                    "academic_offering_specialisation.is_deleted" => 0,
                    "programme_specialisation_type.is_active" => 1,
                    "programme_specialisation_type.is_deleted" => 0,
                    "programme_specialisation_type.name" => "General Specialisation"
                ]
            )
            ->all();

        if (empty($academicOfferingSpecialisations) == false) {
            $specialisations[""] = "Select specialisation...";

            foreach ($academicOfferingSpecialisations as $academicOfferingSpecialisation) {
                $programmeSpecialisation =
                    ProgrammeSpecialisation::find()
                    ->where([
                        "id" => $academicOfferingSpecialisation->programme_specialisation_id,
                        "is_active" => 1,
                        "is_deleted" => 0
                    ])
                    ->one();
                if ($programmeSpecialisation == false) {
                    throw new ModelNotFoundException(
                        "Programme specialisation model not found"
                    );
                } else {
                    $specialisations[$academicOfferingSpecialisation->id] =
                        $programmeSpecialisation->name;
                }
            }
        }
        return $specialisations;
    }


    /**
     * Returns associative array of majors offered in a particular academic
     * offering.
     * ["AcademicOfferingSpecialisation->id" => "ProgrammeSpecification->name']
     *
     * @param integer $academicOfferingId
     * @return array
     */
    private static function getMajors($academicOfferingId)
    {
        $majors = array();

        $academicOfferingSpecialisations =
            AcademicOfferingSpecialisation::find()
            ->innerJoin(
                "programme_specialisation_type",
                "`academic_offering_specialisation`.`programme_specialisation_type_id` = `programme_specialisation_type`.`id`"
            )
            ->where(
                [
                    "academic_offering_specialisation.academic_offering_id" => $academicOfferingId,
                    "academic_offering_specialisation.is_deleted" => 0,
                    "programme_specialisation_type.is_active" => 1,
                    "programme_specialisation_type.is_deleted" => 0,
                    "programme_specialisation_type.name" => "Major"
                ]
            )
            ->all();
        if (empty($academicOfferingSpecialisations) == false) {
            $majors[""] = "Select major ...";

            foreach ($academicOfferingSpecialisations as $academicOfferingSpecialisation) {
                $programmeSpecialisation =
                    ProgrammeSpecialisation::find()
                    ->where([
                        "id" => $academicOfferingSpecialisation->programme_specialisation_id,
                        "is_active" => 1,
                        "is_deleted" => 0
                    ])
                    ->one();
                if ($programmeSpecialisation == false) {
                    throw new ModelNotFoundException(
                        "Programme specialisation model not found"
                    );
                } else {
                    $majors[$academicOfferingSpecialisation->id] =
                        $programmeSpecialisation->name;
                }
            }
        }
        return $majors;
    }


    /**
     * Returns associative array of minors offered in a particular academic
     *  offering.
     * ["AcademicOfferingSpecialisation->id" => "ProgrammeSpecification->name']
     *
     * @param integer $academicOfferingId
     * @return array
     */
    private static function getMinors($academicOfferingId)
    {
        $minors = array();

        $academicOfferingSpecialisations =
            AcademicOfferingSpecialisation::find()
            ->innerJoin(
                "programme_specialisation_type",
                "`academic_offering_specialisation`.`programme_specialisation_type_id` = `programme_specialisation_type`.`id`"
            )
            ->where(
                [
                    "academic_offering_specialisation.academic_offering_id" => $academicOfferingId,
                    "academic_offering_specialisation.is_deleted" => 0,
                    "programme_specialisation_type.is_active" => 1,
                    "programme_specialisation_type.is_deleted" => 0,
                    "programme_specialisation_type.name" => "Minor"
                ]
            )
            ->all();
        if (empty($academicOfferingSpecialisations) == false) {
            $minors[""] = "Select minor ...";

            foreach ($academicOfferingSpecialisations as $academicOfferingSpecialisation) {
                $programmeSpecialisation =
                    ProgrammeSpecialisation::find()
                    ->where([
                        "id" => $academicOfferingSpecialisation->programme_specialisation_id,
                        "is_active" => 1,
                        "is_deleted" => 0
                    ])
                    ->one();
                if ($programmeSpecialisation == false) {
                    throw new ModelNotFoundException(
                        "Programme specialisation model not found"
                    );
                } else {
                    $minors[$academicOfferingSpecialisation->id] =
                        $programmeSpecialisation->name;
                }
            }
        }
        return $minors;
    }
}
