<?php

namespace  app\subcomponents\students\UseCases\ReEnrolment;

use Yii;
use common\models\AcademicOfferingModel;
use common\models\Application;
use common\models\ApplicationPeriod;
use common\models\ApplicationCapesubject;
use common\models\AcademicOfferingSpecialisation;
use common\models\ApplicationSpecialisation;
use common\models\ProgrammeCatalog;
use common\models\Offer;
use common\models\Error;
use common\models\ReEnrolment;
use common\models\Result;
use common\models\StudentRegistration;

class ReEnrolmentUseCase
{
    public function __construct()
    {
    }

    public function handle($request)
    {
        if ($this->programmeChoiceIsCape($request) == true) {
            return $this->processCapeProgrammeChoice($request);
        } elseif (
            $this->programmeChoiceIsAssociateDegreeWithSpecialisations($request)
            == true
        ) {
            return $this->processAssociateDegreeWithSpecialisationsProgrammeChoice(
                $request
            );
        } elseif (
            $this->programmeChoiceIsAssociateDegreeWithoutSpecialisations($request)
            == true
        ) {
            return $this->processAssociateDegreeWithoutSpecialisationsProgrammeChoice(
                $request
            );
        } else {
            return Result::failure(
                new Error("Programme choice cannot be processed.")
            );
        }
    }

    private function createReEnrollmentRecord(
        $personnelId,
        $userId,
        $detailsForm,
        $previousRegistration,
        $newRegistration
    ) {
        $model = new ReEnrolment();
        $model->user_id = $userId;
        $model->personnel_id = $personnelId;
        $model->previous_student_registration_id =
            $previousRegistration->studentregistrationid;
        $model->new_student_registration_id =
            $newRegistration->studentregistrationid;
        $model->comments = $detailsForm->details;
        $model->date = date("Y-m-d");
        $model->is_deleted = 0;
        if ($model->save() == false) {
            return Result::failure(
                new Error("Re-enrolment record creation failed.")
            );
        }
        return Result::success($model);
    }

    private function programmeChoiceIsCape($request)
    {
        if ($request->hasCapeEntry == true) {
            return true;
        }
        return false;
    }
    private function processCapeProgrammeChoice($request)
    {
        $application =
            $this->createApplicationRecord(
                $request->userId,
                $request->academicOfferingId
            );
        if ($application->isFailure()) {
            return $application;
        }

        $offer =
            $this->createOfferRecord(
                $application->getValue(),
                $request->personnelId
            );
        if ($offer->isFailure()) {
            return $offer;
        }

        $archivedStudentRegistration =
            $this->archiveStudentRegistrationRecord(
                $request->currentStudentRegistration
            );
        if ($archivedStudentRegistration->isFailure()) {
            return $archivedStudentRegistration;
        }

        $newStudentRegistration =
            $this->createStudentRegistrationRecord(
                $application->getValue(),
                $offer->getValue()
            );
        if ($newStudentRegistration->isFailure()) {
            return $newStudentRegistration;
        }

        if ($request->hasCapeEntry == true) {
            $capeProcessingFeedback = $this->processCapeSelection(
                $request->applicationCapesubjectForms,
                $application->getValue()
            );
        }
        if ($capeProcessingFeedback->isFailure()) {
            return $capeProcessingFeedback;
        }

        $reEnrolment = $this->createReEnrollmentRecord(
            $request->personnelId,
            $request->userId,
            $request->detailsForm,
            $request->currentStudentRegistration,
            $newStudentRegistration->getValue()
        );
        if ($reEnrolment->isFailure()) {
            return $reEnrolment;
        }

        return Result::success($newStudentRegistration->getValue());
    }

    private function programmeChoiceIsAssociateDegreeWithoutSpecialisations(
        $request
    ) {
        if (
            $request->hasCapeEntry == false
            && $request->hasSpecialisationEntry == false
        ) {
            return true;
        }
        return false;
    }
    private function processAssociateDegreeWithoutSpecialisationsProgrammeChoice(
        $request
    ) {
        $application =
            $this->createApplicationRecord(
                $request->userId,
                $request->academicOfferingId
            );
        if ($application->isFailure()) {
            return $application;
        }

        $offer =
            $this->createOfferRecord(
                $application->getValue(),
                $request->personnelId
            );
        if ($offer->isFailure()) {
            return $offer;
        }

        $archivedStudentRegistration =
            $this->archiveStudentRegistrationRecord(
                $request->currentStudentRegistration
            );
        if ($archivedStudentRegistration->isFailure()) {
            return $archivedStudentRegistration;
        }

        $newStudentRegistration =
            $this->createStudentRegistrationRecord(
                $application->getValue(),
                $offer->getValue()
            );
        if ($newStudentRegistration->isFailure()) {
            return $newStudentRegistration;
        }

        $reEnrolment = $this->createReEnrollmentRecord(
            $request->personnelId,
            $request->userId,
            $request->detailsForm,
            $request->currentStudentRegistration,
            $newStudentRegistration->getValue()
        );
        if ($reEnrolment->isFailure()) {
            return $reEnrolment;
        }

        return Result::success($newStudentRegistration->getValue());
    }

    private function programmeChoiceIsAssociateDegreeWithSpecialisations(
        $request
    ) {
        if (
            $request->hasCapeEntry == false
            && $request->hasSpecialisationEntry == true
        ) {
            return true;
        }
        return false;
    }
    private function processAssociateDegreeWithSpecialisationsProgrammeChoice(
        $request
    ) {
        $application =
            $this->createApplicationRecord(
                $request->userId,
                $request->academicOfferingId
            );
        if ($application->isFailure()) {
            return $application;
        }

        $offer =
            $this->createOfferRecord(
                $application->getValue(),
                $request->personnelId
            );
        if ($offer->isFailure()) {
            return $offer;
        }

        $archivedStudentRegistration =
            $this->archiveStudentRegistrationRecord(
                $request->currentStudentRegistration
            );
        if ($archivedStudentRegistration->isFailure()) {
            return $archivedStudentRegistration;
        }

        $newStudentRegistration =
            $this->createStudentRegistrationRecord(
                $application->getValue(),
                $offer->getValue()
            );
        if ($newStudentRegistration->isFailure()) {
            return $newStudentRegistration;
        }

        if ($request->hasSpecialisationEntry == true) {
            $specialisation = $this->createSpecialisation(
                $request,
                $application->getValue()
            );
            if ($specialisation->isFailure()) {
                return $specialisation;
            }
        }

        $reEnrolment = $this->createReEnrollmentRecord(
            $request->personnelId,
            $request->userId,
            $request->detailsForm,
            $request->currentStudentRegistration,
            $newStudentRegistration->getValue()
        );
        if ($reEnrolment->isFailure()) {
            return $reEnrolment;
        }

        return Result::success($newStudentRegistration->getValue());
    }

    private function archiveStudentRegistrationRecord($studentRegistration)
    {
        $studentRegistration->isactive = 0;
        $studentRegistration->isdeleted = 0;
        if ($studentRegistration->save() == false) {
            return Result::failure(
                new Error("The archiving of current enrolment failed.")
            );
        }
        return Result::success($studentRegistration);
    }

    private function getApplicationOrdering($userId)
    {
        $applications = Application::find()
            ->where([
                "personid" => $userId,
                "isactive" => 1,
                "isdeleted" => 0
            ])
            ->orderBy('ordering DESC')
            ->all();
        if (empty($applications)) {
            Result::failure(new Error("Applications not found."));
        }
        return Result::success($applications[0]->ordering + 1);
    }

    private function createApplicationRecord(
        $userId,
        $academicOfferingId
    ) {
        $application = new Application();
        $application->personid = $userId;

        $divisionId = $this->getAcademicOfferingDivisionId($academicOfferingId);
        if ($divisionId->isFailure()) {
            return $divisionId;
        }
        $application->divisionid = $divisionId->getValue();

        $application->academicofferingid = $academicOfferingId;
        $application->applicationstatusid = 9;
        $application->applicationtimestamp = date('Y-m-d H:i:s');
        $application->submissiontimestamp = date('Y-m-d H:i:s');

        $nextOrdering = $this->getApplicationOrdering($userId);
        if ($nextOrdering->isFailure()) {
            return $nextOrdering;
        }
        $application->ordering = $nextOrdering->getValue();
        $application->ipaddress = Yii::$app->request->getUserIP();
        $application->browseragent = Yii::$app->request->getUserAgent();
        $application->isactive = 1;
        $application->isdeleted = 0;

        if ($application->save() == false) {
            return Result::failure(
                new Error("Application record creation failed.")
            );
        }
        return Result::success($application);
    }


    private function createOfferRecord($application, $personnelId)
    {
        $offer = new Offer();
        $offer->applicationid = $application->applicationid;
        $offer->issuedby = $personnelId;
        $offer->issuedate = date('Y-m-d');
        $offer->offertypeid = 1;
        $offer->ispublished = 1;
        $offer->isactive = 1;
        $offer->isdeleted = 0;

        if ($offer->save() == false) {
            return Result::failure(new Error("Offer record creation failed."));
        }
        return Result::success($offer);
    }

    private function getRegistrationTypeId($academicOfferingId)
    {
        $programmeCatalog =
            ProgrammeCatalog::find()
            ->innerJoin(
                'academic_offering',
                '`programme_catalog`.`programmecatalogid` = `academic_offering`.`programmecatalogid`'
            )
            ->where([
                'programme_catalog.isdeleted' => 0,
                'academic_offering.academicofferingid' => $academicOfferingId,
                'academic_offering.isdeleted' => 0
            ])
            ->all();
        if (empty($programmeCatalog)) {
            return Result::failure(new Error("Programme type not found."));
        }

        return Result::success($programmeCatalog[0]->programmetypeid);
    }

    private function createStudentRegistrationRecord($application, $offer)
    {
        $registration = new StudentRegistration();
        $registration->offerid = $offer->offerid;
        $registration->personid = $application->personid;
        $registration->academicofferingid = $application->academicofferingid;

        $programmeType =
            $this->getRegistrationTypeId($application->academicofferingid);
        if ($programmeType->isFailure()) {
            return $programmeType;
        }
        $registration->registrationtypeid = $programmeType->getValue();

        $registration->currentlevel = 1;
        $registration->registrationdate = date('Y-m-d');
        if ($registration->save() == false) {
            return Result::failure(
                new Error("Registration record creation failed.")
            );
        }
        return Result::success($registration);
    }

    private function capeSubjectSelectionValid($applicationCapesubjects)
    {
        $minimumAllowed = 2;
        $maximumAllowed = 5;
        $selected = 0;
        if (!empty($applicationCapesubjects)) {
            return false;
        }

        foreach ($applicationCapesubjects as $record) {
            if ($record->capesubjectid != 0)   //if valid subject is selected
            {
                $selected++;
            }
        }

        if ($selected >= $minimumAllowed && $selected <= $maximumAllowed) {
            return true;
        }

        return false;
    }

    private function applicationSubjectSelected($applicationCapeSubject)
    {
        if (
            $applicationCapeSubject->capesubjectid != 0
            && $applicationCapeSubject->applicationid != 0
        ) {
            return true;
        }
        return false;
    }

    private function createApplicationCapesubject($capeSubject, $application)
    {
        $subjectSelection = new ApplicationCapesubject();
        $subjectSelection->capesubjectid = $capeSubject->capesubjectid;
        $subjectSelection->applicationid = $application->applicationId;
        $subjectSelection->isactive = 1;
        $subjectSelection->isdeleted = 0;
        if ($subjectSelection->save() == false) {
            return Result::failure(new Error("Cape subject selection failure"));
        }
        return Result::success($subjectSelection);
    }

    private function processCapeSelection(
        $applicationCapeSubjectForms,
        $application
    ) {
        $selectedCapeSubjects = array();
        foreach ($applicationCapeSubjectForms as $entry) {
            $entry->applicationid = $application->applicationid;      //updates applicationid

            if ($this->applicationSubjectSelected($entry) == true) {
                $entry->isactive = 1;
                $entry->isdeleted = 0;
                if ($entry->save() == false) {
                    return Result::failure(
                        new Error("Cape subject selection failed.")
                    );
                } else {
                    array_push($selectedCapeSubjects, $entry);
                }
            }
        }
        if (empty($selectedCapeSubjects)) {
            return Result::failure(
                new Error("You must select at least one CAPE subject.")
            );
        }
        return Result::success($selectedCapeSubjects);
    }

    private function createSpecialisation($request, $application)
    {
        if ($this->findIdenticalMajorMinorSelection(
            $request->programmeSpecialisationChoiceForm
        ) == true) {
            return Result::failure(
                new Error("Major and minor must be different")
            );
        }

        if ($this->hasIncompleteMandatorySpecialisations(
            $request->programmeSpecialisationChoiceForm,
            $request
        ) == true) {
            return Result::failure(
                new Error("All mandatory specialisations must be entered.")
            );
        }

        if (
            $request->programmeSpecialisationChoiceForm->general_specialisation_id == true
            || $request->programmeSpecialisationChoiceForm->major_id == true
            || $request->programmeSpecialisationChoiceForm->minor_id == true
        ) {
            $specialisations =
                $this->processSpecialisations($request, $application);

            foreach ($specialisations as $specialisation) {
                $specialisation->save();
            }
        }
        return Result::success($specialisations);
    }

    private function findIdenticalMajorMinorSelection(
        $programmeSpecialisationChoiceForm
    ) {
        if (
            $programmeSpecialisationChoiceForm->major_id == true
            &&  $programmeSpecialisationChoiceForm->minor_id == true
        ) {
            $majorAcademicOfferingSpecification =
                AcademicOfferingSpecialisation::find()
                ->where(["id" => $programmeSpecialisationChoiceForm->major_id, "is_deleted" => 0])
                ->one();

            $minorAcademicOfferingSpecification =
                AcademicOfferingSpecialisation::find()
                ->where(["id" => $programmeSpecialisationChoiceForm->minor_id, "is_deleted" => 0])
                ->one();

            if (
                $majorAcademicOfferingSpecification->programme_specialisation_id
                == $minorAcademicOfferingSpecification->programme_specialisation_id
            ) {
                return true;
            }
        }
        return false;
    }

    private function hasIncompleteMandatorySpecialisations(
        $programmeSpecialisationChoiceForm,
        $request
    ) {
        if (
            $request->isGeneralSpecialisationMandatory == true
            && $programmeSpecialisationChoiceForm->general_specialisation_id == false
        ) {
            return true;
        } elseif (
            $request->isMajorMandatory == true
            && $programmeSpecialisationChoiceForm->major_id == false
        ) {
            return true;
        } elseif (
            $request->isMinorMandatory == true
            && $programmeSpecialisationChoiceForm->minor_id == false
        ) {
            return true;
        }
        return false;
    }

    private function processSpecialisations($request, $application)
    {
        $specialisations = array();

        if ($request->programmeSpecialisationChoiceForm->general_specialisation_id == true) {
            $generalSpecialisation = new ApplicationSpecialisation();
            $generalSpecialisation->id = uniqid();
            $generalSpecialisation->application_id = $application->applicationid;

            $generalSpecialisation->academic_offering_specialisation_id =
                $request->programmeSpecialisationChoiceForm->general_specialisation_id;

            $generalSpecialisation->is_deleted = 0;
            if ($generalSpecialisation->validate() == true) {
                array_push($specialisations, $generalSpecialisation);
            } else {
                return Result::failure(
                    new Error("Error generating general specialisation model")
                );
            }
        }

        if ($request->programmeSpecialisationChoiceForm->major_id == true) {
            $major = new ApplicationSpecialisation();
            $major->id = uniqid();
            $major->application_id = $application->applicationid;

            $major->academic_offering_specialisation_id =
                $request->programmeSpecialisationChoiceForm->major_id;

            $major->is_deleted = 0;
            if ($major->validate() == true) {
                array_push($specialisations, $major);
            } else {
                return Result::failure(
                    new Error("Error generating major specialisation model")
                );
            }
        }

        if ($request->programmeSpecialisationChoiceForm->minor_id == true) {
            $minor = new ApplicationSpecialisation();
            $minor->id = uniqid();
            $minor->application_id = $application->applicationid;

            $minor->academic_offering_specialisation_id =
                $request->programmeSpecialisationChoiceForm->minor_id;

            $minor->is_deleted = 0;
            if ($minor->validate() == true) {
                array_push($specialisations, $minor);
            } else {
                return Result::failure(
                    new Error("Error generating minor specialisation model")
                );
            }
        }
        return $specialisations;
    }

    private function getAcademicOfferingDivisionId($academicOfferingId)
    {
        $applicationPeriods =
            ApplicationPeriod::find()
            ->innerJoin(
                "academic_offering",
                "`application_period`.`applicationperiodid` = `academic_offering`.`applicationperiodid`"
            )
            ->where([
                "application_period.isdeleted" => 0,
                "academic_offering.academicofferingid" => $academicOfferingId,
                "academic_offering.isactive" => 1,
                "academic_offering.isdeleted" => 0
            ])
            ->all();
        if (empty($applicationPeriods)) {
            return Result::failure(
                new Error("Error determining division")
            );
        }
        return Result::success($applicationPeriods[0]->divisionid);
    }
}
