<?php

namespace app\subcomponents\graduation\models;

use common\models\GraduationPackageChangeRequest;
use common\models\GraduationPackageChangeRequestStatus;
use common\models\BillingCharge;
use yii\data\ArrayDataProvider;
use yii\helpers\Html;
use yii\helpers\Url;


class PackageChangeRequestsListingViewModel
{
    public $pageTitle;
    public $startDate;
    public $endDate;
    public $dataProvider;

    private function __construct(
        $pageTitle,
        $startDate,
        $endDate,
        $dataProvider
    ) {
        $this->pageTitle = $pageTitle;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->dataProvider = $dataProvider;
    }


    private static function getAllRequests()
    {
        return GraduationPackageChangeRequest::find()
            ->where(["is_deleted" => 0])
            ->all();
    }


    public static function convertGraduationPackageChangeRequestsIntoAssociativeArray(
        $request
    ) {
        $record = array();
        $record["id"] = $request->id;
        $record["dateCreated"] = date_format(new \DateTime($request->date_created), "F j, Y");
        $record["username"] = $request->username;
        $record["firstName"] = $request->first_name;
        $record["lastName"] = $request->last_name;

        if ($request->graduation_package_change_request_type_id == 1) {
            $record["type"] = "Late Selection";
        } else {
            $record["type"] = "Post Payment Request";
        }

        $record["status"] =
            GraduationPackageChangeRequestStatus::find()
            ->where([
                "id" => $request->graduation_package_change_request_status_id,
                "is_deleted" => 0
            ])
            ->one()
            ->name;

        if ($request->date_reviewed == true) {
            $record["dateReviewed"] =
                date_format(new \DateTime($request->date_reviewed), "F j, Y");
        } else {
            $record["dateReviewed"] = "";
        }

        if ($request->date_paid == true) {
            $record["datePaid"] =
                date_format(new \DateTime($request->date_paid), "F j, Y");
        } else {
            $record["datePaid"] = "";
        }

        $record["reviewOptions"] = self::generateReviewOptions($request);

        $currentPackage =
            BillingCharge::find()
            ->where([
                "id" => $request->current_package_id,
                "is_active" => 1,
                "is_deleted" => 0
            ])
            ->one();

        if ($currentPackage == true) {
            $currentPackageName = $currentPackage->getBillingType()->one()->name;
            $currentPackageDescription =
                $currentPackage->getBillingType()->one()->description;

            $record["currentPackage"] =
                "{$currentPackageName} <br/> {$currentPackageDescription}";
        } else {
            $record["currentPackage"] = "";
        }

        $requestedPackage =
            BillingCharge::find()
            ->where([
                "id" => $request->proposed_package_id,
                "is_active" => 1,
                "is_deleted" => 0
            ])
            ->one();
        $requestedPackageName = $requestedPackage->getBillingType()->one()->name;
        $requestedPackageDescription =
            $requestedPackage->getBillingType()->one()->description;

        $record["requestedPackage"] =
            "{$requestedPackageName} <br/> {$requestedPackageDescription}";

        return $record;
    }


    private static function generateReviewOptions($request)
    {
        $options = "";

        if ($request->graduation_package_change_request_status_id == 1) {
            $approveButton =
                Html::a(
                    "Approve",
                    Url::toRoute([
                        "package-change-requests/approve",
                        "id" => $request->id
                    ]),
                    [
                        "class" => "btn btn-sm btn-success",
                        "style" => "margin:5px; width:75px"
                    ]
                );

            $options .= $approveButton;

            $rejectButton =
                Html::a(
                    "Reject",
                    Url::toRoute([
                        "package-change-requests/reject",
                        "id" => $request->id
                    ]),
                    [
                        "class" => "btn btn-sm btn-danger",
                        "style" => "margin:5px; width:75px"
                    ]
                );

            $options .= $rejectButton;
        }

        return $options;
    }

    private static function preparePackageChangeRequestsListing($requests)
    {
        $dataSet = array();
        foreach ($requests as $request) {
            $dataSet[] =
                self::convertGraduationPackageChangeRequestsIntoAssociativeArray(
                    $request
                );
        }
        return $dataSet;
    }


    public static function create($status, $startDate = null, $endDate = null)
    {
        if ($status == "view-all") {
            $pageTitle = "All Package Change Requests";

            $requests = self::getAllRequests();

            $dataProvider =
                new ArrayDataProvider(
                    [
                        "allModels" =>
                        self::preparePackageChangeRequestsListing($requests),
                        "pagination" => ["pageSize" => 100],
                        "sort" => [
                            "defaultOrder" => ["dateCreated" => SORT_DESC],
                            "attributes" => [
                                "dateCreated",
                                "username",
                                "lastName",
                                "status"
                            ]
                        ]
                    ]
                );

            return new PackageChangeRequestsListingViewModel(
                $pageTitle,
                $startDate,
                $endDate,
                $dataProvider
            );
        }
    }
}
