<?php

namespace app\subcomponents\bursary\views\viewmodels\registryandlibraryservices;

use frontend\core\domain\exceptions\RequestCreationException;

/**
 * Factory class for PaymentPreview class
 */
class PaymentPreviewFactory
{
    private function __construct()
    {
    }


    /**
     * Build RceiptPreview and BillingPreview models
     *
     * @param [type] $userId
     * @param [type] $registryAndLibraryFeesViewModel
     * @param [type] $employeeRepository
     * @return void
     */
    public static function create(
        $userId,
        $registryAndLibraryFeesViewModel,
        $employeeRepository
    ) {
        $receiptForm = $registryAndLibraryFeesViewModel->receiptForm;
        $billingForms = $registryAndLibraryFeesViewModel->billingForms;

        if (self::validateUserInput(
            $receiptForm,
            $billingForms
        ) == true) {
            $receiptPreview =
                new ReceiptPreview(
                    $userId,
                    $registryAndLibraryFeesViewModel->studentRegistrationId,
                    $receiptForm->username,
                    $receiptForm->fullName,
                    $receiptForm->datePaid,
                    $receiptForm->notes,
                    $receiptForm->chequeNumber,

                    self::getFormattedPaidDate(
                        $receiptForm->datePaid
                    ),

                    self::getPaymentMethod(
                        $receiptForm,
                        $billingForms
                    ),

                    self::getOperatorCode($employeeRepository, $userId),
                    self::calculateBillingsTotal($billingForms)
                );

            $billingsPreview = array();

            foreach ($billingForms as $billingForm) {
                if ($billingForm->isActive == true) {
                }
                $billingPreview =
                    new BillingPreview(
                        $billingForm->isActive,
                        $billingForm->cost,
                        $billingForm->quantity,
                        $billingForm->paymentMethod,
                        $billingForm->billingChargeId,
                        $billingForm->feeName
                    );
                array_push($billingsPreview, $billingPreview);
            }

            return new PaymentPreview($receiptPreview, $billingsPreview);
        }
    }


    /**
     * Return date in for month day, year e.g. => November 28, 2022
     *
     * @param string $datePaid
     * @return string
     */
    private static function getFormattedPaidDate($datePaid)
    {
        return date_format(new \DateTime($datePaid), "F j, Y");
    }


    /**
     * Generates payment method summary
     *
     * @param ReceiptForm $receiptForm
     * @param array $billingForms
     * @return string
     */
    private static function getPaymentMethod($receiptForm, $billingForms)
    {
        $uniquePaymentMethods = array();
        foreach ($billingForms as $billing) {
            if (in_array(
                $billing->paymentMethod,
                $uniquePaymentMethods
            ) == false) {
                array_push($uniquePaymentMethods, $billing->paymentMethod);
            }
        }

        if (count($uniquePaymentMethods) == 1) {
            if ($billingForms[0]->paymentMethod === "Cheque") {
                return "Cheque#{$receiptForm->chequeNumber}";
            } else {
                return $uniquePaymentMethods[0];
            }
        } elseif (count($uniquePaymentMethods) > 1) {
            $paymentMethods = "";
            foreach ($uniquePaymentMethods as $paymentMethod) {
                if (
                    $paymentMethod === "Cheque"
                    && $receiptForm->chequeNumber == true
                ) {
                    $paymentMethods .= "Cheque#{$receiptForm->chequeNumber},";
                } elseif ($paymentMethod === "Cheque" && $receiptForm->chequeNumber == false) {
                    $paymentMethods .= "Cheque,";
                } else {
                    $paymentMethods .= "{$paymentMethod},";
                }
            }
            if ($paymentMethods != "") {
                $paymentMethods = substr($paymentMethods, 0, -1); // remove trailing comma
            }
            return $paymentMethods;
        } else {
            return "Combined";
        }
    }


    /**
     * Generates cashier code for employee
     *
     * @param employeeRepository $employeeRepository
     * @param integer $userId
     * @return string
     */
    private static function getOperatorCode($employeeRepository, $userId)
    {
        $employee = $employeeRepository->getEmployeeByPersonId($userId);
        $last = substr($employee->lastname, 0, 3);
        $first = substr($employee->firstname, 0, 3);
        $code = "{$last}{$first}";
        return strtoupper($code);
    }


    /**
     * Returns true if at lease one valid billing exists
     *
     * @param array $billingForms
     * @return boolean
     */
    private static function requestHasAtLeastOneValidBilling($billingForms)
    {
        if (!empty($billingForms)) {
            foreach ($billingForms as $billingForm) {
                if ($billingForm->validateModel() == true) {
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * Returns true if any billing items indicate cheque payment
     *
     * @param array $billingForms
     * @return boolean
     */
    private static function anyBillingsPaidByCheque($billingForms)
    {
        if (!empty($billingForms)) {
            foreach ($billingForms as $billing) {
                if (
                    $billing->validateModel() == true
                    && $billing->paymentMethod == "Cheque"
                ) {
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * PErforms summation of billings
     *
     * @param array $billingForms
     * @return float
     */
    private static function calculateBillingsTotal($billingForms)
    {
        $total = 0;
        if (!empty($billingForms)) {
            foreach ($billingForms as $billing) {
                if ($billing->validateModel() == true) {
                    $total += ($billing->cost * $billing->quantity);
                }
            }
        }
        return number_format($total, 2);
    }


    /**
     * Performs validation checks on user payment input
     *
     * @param ReceiptForm $receiptForm
     * @param array $billingForms
     * @return boolean
     */
    private static function validateUserInput(
        $receiptForm,
        $billingForms
    ) {
        if (self::requestHasAtLeastOneValidBilling(
            $billingForms
        ) == false) {
            throw new RequestCreationException(
                "No fees were selected for payment."
            );
        } elseif (
            $receiptForm->payUsingCheque == true
            && $receiptForm->chequeNumber == false
        ) {
            throw new RequestCreationException("Cheque number missing.");
        } elseif (
            $receiptForm->payUsingCheque == true
            && self::anyBillingsPaidByCheque($billingForms)
            == false
        ) {
            throw new RequestCreationException(
                "You must have at least one billing indicating cheque payment."
            );
        } elseif (
            $receiptForm->payUsingCheque == false
            && self::anyBillingsPaidByCheque($billingForms) == true
        ) {
            throw new RequestCreationException(
                "You indicated none cheque payment but subsequently added billing with cheque."
            );
        }
        return true;
    }
}
