<?php

namespace app\subcomponents\bursary\models;

use common\models\Billing;
use common\models\BillingCharge;
use common\models\GraduationPackageChangeRequest;
use common\models\GraduationPackageChangeRequestStatus;
use common\models\GraduationSurvey;
use common\models\Receipt;
use yii\data\ArrayDataProvider;
use yii\helpers\Html;
use yii\helpers\Url;

class ViewGraduationPaymentsViewModel
{
    public $pageTitle;
    public $username;
    public $studentRegistrationId;
    public $studentFullName;
    public $graduationPayments;
    public $graduationPaymentOptions;
    public $pendingChangeRequestSummary;
    public $changeRequestHistory;

    public function __construct(
        $pageTitle,
        $username,
        $studentRegistrationId,
        $studentFullName,
        $graduationPayments,
        $graduationPaymentOptions,
        // $pendingChangeRequestSummary,
        $changeRequestHistory
    ) {
        $this->pageTitle = $pageTitle;
        $this->username = $username;
        $this->studentRegistrationId = $studentRegistrationId;
        $this->studentFullName = $studentFullName;
        $this->graduationPayments = $graduationPayments;
        $this->graduationPaymentOptions = $graduationPaymentOptions;
        // $this->pendingChangeRequestSummary = $pendingChangeRequestSummary;
        $this->changeRequestHistory = $changeRequestHistory;
    }


    public static function create(
        $username,
        $studentRegistrationId,
        $studentFullName
    ) {
        $pageTitle = "Graduation Payments";
        $graduationPayments = array();

        $graduationReceipts =
            Receipt::find()
            ->innerJoin(
                'billing',
                '`billing`.`receipt_id` = `receipt`.`id`'
            )
            ->innerJoin(
                'billing_charge',
                '`billing`.`billing_charge_id` = `billing_charge`.`id`'
            )
            ->innerJoin(
                'billing_type',
                '`billing_charge`.`billing_type_id` = `billing_type`.`id`'
            )->innerJoin(
                'billing_category',
                '`billing_type`.`billing_category_id` = `billing_category`.`id`'
            )
            ->where([
                "receipt.student_registration_id" => $studentRegistrationId,
                "receipt.is_active" => 1,
                "receipt.is_deleted" => 0,
                "billing_category.name" => "Graduation Services"
            ])
            ->all();
        $graduationPaymentModels =
            self::createGraduationPayments($graduationReceipts);

        $graduationPayments =
            self::buildPaymentsDataProvider($graduationPaymentModels);

        $graduationSurvey =
            GraduationSurvey::find()
            ->where([
                "student_registration_id" => $studentRegistrationId,
                "is_deleted" => 0
            ])
            ->one();

        // $pendingChangeRequestSummary = null;
        // $activeChangeRequest =
        //     GraduationPackageChangeRequest::find()
        //     ->where([
        //         "student_registration_id" => $studentRegistrationId,
        //         "graduation_package_change_request_status_id" => [1, 2],
        //         "is_deleted" => 0
        //     ])
        //     ->one();

        $activeChangeRequest = NULL;
        $activeChangeRequests =
            GraduationPackageChangeRequest::find()
            ->where([
                "student_registration_id" => $studentRegistrationId,
                "graduation_package_change_request_status_id" => [1, 2],
                "is_deleted" => 0
            ])
            ->orderBy("id DESC")
            ->all();
        if (!empty($activeChangeRequests)) {
            $activeChangeRequest = $activeChangeRequests[0];    // most recent change request
        }

        // if (
        //     $activeChangeRequest == true
        //     && $activeChangeRequest->graduation_package_change_request_status_id == 1
        // ) {
        //     $pendingChangeRequestSummary = GraduationPackageChangeRequestSummary::create($activeChangeRequest)->toString();
        // }


        $graduationBillings =
            Billing::find()
            ->innerJoin(
                'billing_charge',
                '`billing`.`billing_charge_id` = `billing_charge`.`id`'
            )
            ->innerJoin(
                'billing_type',
                '`billing_charge`.`billing_type_id` = `billing_type`.`id`'
            )->innerJoin(
                'billing_category',
                '`billing_type`.`billing_category_id` = `billing_category`.`id`'
            )
            ->where([
                "billing.student_registration_id" => $studentRegistrationId,
                "billing.is_active" => 1,
                "billing.is_deleted" => 0,
                "billing_category.name" => "Graduation Services"
            ])
            ->all();

        $graduationPaymentOptions =
            self::getPaymentOptions(
                $graduationBillings,
                $studentRegistrationId,
                $graduationSurvey,
                $activeChangeRequest
            );

        $changeRequests = self::getChangeRequests($studentRegistrationId);
        $changeRequestDataProvider =
            new ArrayDataProvider(
                [
                    "allModels" => self::preparePackageChangeRequestsListing($changeRequests),
                    "pagination" => ["pageSize" => 20]
                ]
            );

        return new ViewGraduationPaymentsViewModel(
            $pageTitle,
            $username,
            $studentRegistrationId,
            $studentFullName,
            $graduationPayments,
            $graduationPaymentOptions,
            // $pendingChangeRequestSummary,
            $changeRequestDataProvider
        );
    }

    private static function createGraduationPayments($receipts)
    {
        if ($receipts == false) {
            return array();
        }

        $graduationPayments = array();
        foreach ($receipts as $receipt) {
            $billing = Billing::find()->where(["receipt_id" => $receipt->id])->one();
            array_push(
                $graduationPayments,
                GraduationPayment::create($receipt, $billing)
            );
        }

        return $graduationPayments;
    }


    private static function buildPaymentsDataProvider($graduationPayments)
    {
        $rows = array();
        if (!empty($graduationPayments)) {
            foreach ($graduationPayments as $graduationPayment) {
                $row["username"] = $graduationPayment->username;
                $row["receiptId"] = $graduationPayment->receiptId;
                $row["receiptNumber"] = $graduationPayment->receiptNumber;
                $row["billingChargeId"] = $graduationPayment->billingChargeId;
                $row["billingTypeName"] = $graduationPayment->billingTypeName;
                $row["billingTypeDescription"] = $graduationPayment->billingTypeDescription;
                $row["amount"] = $graduationPayment->amount;
                $row["datePaid"] = $graduationPayment->datePaid;
                array_push($rows, $row);
            }
        }

        return new ArrayDataProvider(
            [
                "allModels" => $rows,
                "pagination" => ["pageSize" => 20],
                "sort" => [
                    "defaultOrder" => ["receiptNumber" => SORT_ASC],
                    "attributes" => ["receiptNumber"]
                ]
            ]
        );
    }


    private static function getPaymentOptions(
        $graduationPayments,
        $studentRegistrationId,
        $graduationSurvey,
        $activeChangeRequest
    ) {
        $paymentOptions = array();

        if (
            GraduationOperationValidator::canRegisterAndPayPackage(
                $graduationSurvey,
                $activeChangeRequest,
                $graduationPayments
            ) == true
        ) {
            array_push(
                $paymentOptions,
                self::generateRegisterAndPayPackageButton($studentRegistrationId)
            );
        }

        if (GraduationOperationValidator::canMakePayment($graduationSurvey, $activeChangeRequest, $graduationPayments)) {
            array_push(
                $paymentOptions,
                self::generateMakePackageButton($studentRegistrationId)
            );
        }

        if (GraduationOperationValidator::canCreatePrePaymentChangeRequest($graduationSurvey, $activeChangeRequest, $graduationPayments)) {
            array_push(
                $paymentOptions,
                self::generatePrePaymentPackageRequestButton($studentRegistrationId)
            );
        }

        if (GraduationOperationValidator::canCreatePostPaymentChangeRequest($graduationPayments, $activeChangeRequest)) {
            array_push(
                $paymentOptions,
                self::generatePostPaymentPackageRequestButton($studentRegistrationId)
            );
        }

        return $paymentOptions;
    }


    private static function generateRegisterAndPayPackageButton($studentRegistrationId)
    {
        return Html::a(
            "Make Payment",
            Url::toRoute([
                "graduation-payments/register-and-pay-package",
                "studentRegistrationId" => $studentRegistrationId
            ]),
            [
                "class" => "btn btn-sm btn-success",
                "style" => "margin:5px; width:100px"
            ]
        );
    }


    private static function generateMakePackageButton($studentRegistrationId)
    {
        return Html::a(
            "Make Payment",
            Url::toRoute([
                "graduation-payments/make-payment",
                "studentRegistrationId" => $studentRegistrationId
            ]),
            [
                "class" => "btn btn-sm btn-success",
                "style" => "margin:5px; width:100px"
            ]
        );
    }


    private static function generatePostPaymentPackageRequestButton($studentRegistrationId)
    {
        return Html::a(
            "Post Payment Change Request",
            Url::toRoute([
                "graduation-payments/post-payment-package-modification-request-entry",
                "studentRegistrationId" => $studentRegistrationId
            ]),
            [
                "class" => "btn btn-sm btn-primary",
                "style" => "margin:5px; min-width:100px"
            ]
        );
    }


    // private static function generatePrePaymentPackageRequestButton($studentRegistrationId)
    // {
    //     return Html::a(
    //         "Pre-Payment Change Request",
    //         Url::toRoute([
    //             "graduation-payments/change-package",
    //             "studentRegistrationId" => $studentRegistrationId
    //         ]),
    //         [
    //             "class" => "btn btn-sm btn-primary",
    //             "style" => "margin:5px; min-width:100px"
    //         ]
    //     );
    // }
    private static function generatePrePaymentPackageRequestButton($studentRegistrationId)
    {
        return Html::a(
            "Pre-Payment Change Request",
            Url::toRoute([
                "graduation-payments/pre-payment-package-modification-request-entry",
                "studentRegistrationId" => $studentRegistrationId
            ]),
            [
                "class" => "btn btn-sm btn-primary",
                "style" => "margin:5px; min-width:100px"
            ]
        );
    }

    private static function getChangeRequests($studentRegistrationId)
    {
        return GraduationPackageChangeRequest::find()
            ->where(["student_registration_id" => $studentRegistrationId, "is_deleted" => 0])
            ->all();
    }

    private static function preparePackageChangeRequestsListing($changeRequests)
    {
        $dataSet = array();
        foreach ($changeRequests as $request) {
            $dataSet[] = self::convertGraduationPackageChangeRequestsIntoAssociativeArray($request);
        }
        return $dataSet;
    }


    private static function convertGraduationPackageChangeRequestsIntoAssociativeArray($request)
    {
        $record = array();
        $record["id"] = $request->id;
        $record["dateCreated"] = date_format(new \DateTime($request->date_created), "F j, Y");
        $record["username"] = $request->username;
        $record["firstName"] = $request->first_name;
        $record["lastName"] = $request->last_name;

        $record["status"] =
            GraduationPackageChangeRequestStatus::find()
            ->where([
                "id" => $request->graduation_package_change_request_status_id,
                "is_deleted" => 0
            ])
            ->one()
            ->name;

        if ($request->date_reviewed == true) {
            $record["dateReviewed"] =
                date_format(new \DateTime($request->date_reviewed), "F j, Y");
        } else {
            $record["dateReviewed"] = "";
        }

        $currentPackage =
            BillingCharge::find()
            ->where([
                "id" => $request->current_package_id,
                "is_active" => 1,
                "is_deleted" => 0
            ])
            ->one();

        if ($currentPackage == true) {
            $currentPackageName = $currentPackage->getBillingType()->one()->name;
            $currentPackageDescription =
                $currentPackage->getBillingType()->one()->description;

            $record["currentPackage"] =
                "{$currentPackageName} <br/> {$currentPackageDescription}";
        } else {
            $record["currentPackage"] = "";
        }

        $requestedPackage =
            BillingCharge::find()
            ->where(["id" => $request->proposed_package_id, "is_active" => 1, "is_deleted" => 0])
            ->one();
        $requestedPackageName = $requestedPackage->getBillingType()->one()->name;
        $requestedPackageDescription =
            $requestedPackage->getBillingType()->one()->description;

        $record["requestedPackage"] =
            "{$requestedPackageName} <br/> {$requestedPackageDescription}";

        return $record;
    }
}
