<?php

namespace app\subcomponents\bursary\models;

use common\models\Billing;
use common\models\BillingCharge;
use common\models\BillingModel;
use common\models\Credit;
use common\models\ErrorObject;
use common\models\GraduationSurvey;
use common\models\Receipt;
use common\models\StudentRegistration;
use common\models\UserModel;
use yii\base\Model;


class SingleFeePaymentForm extends Model
{
    public $username;
    public $fullName;
    public $balance;
    public $amountPaid;
    public $paymentMethodId;
    public $yearPaid;
    public $monthPaid;
    public $dayPaid;
    public $customerId;
    public $applicationPeriodId;
    public $billingChargeId;
    public $chequeNumber;

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [
                [
                    "username",
                    "fullName",
                    "balance",
                    "amountPaid",
                    "paymentMethodId",
                    "includeAmendmentFee",
                    "datePaid",
                    "customerId",
                    "applicationPeriodId",
                    "billingChargeId",
                    "yearPaid",
                    "monthPaid",
                    "dayPaid"
                ],
                "required"
            ],
            [["amountPaid", "balance"], "number"],
            [["username", "fullName", "chequeNumber"], "string"],
            [
                [
                    "paymentMethodId",
                    "customerId",
                    "applicationPeriodId",
                    "billingChargeId"
                ],
                "integer"
            ],
            [["yearPaid"], "string", "min" => 4],
            [["yearPaid"], "string", "max" => 4],
            [["monthPaid"], "string", "min" => 2],
            [["monthPaid"], "string", "max" => 2],
            [["dayPaid"], "string", "min" => 2],
            [["dayPaid"], "string", "max" => 2]
        ];
    }


    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            "username" => "ApplicantID",
            "fullName" => "Full Name",
            "balance" => "Balance",
            "amountPaid" => "AmountPaid",
            "paymentMethodId" => "Payment Method",
            "year" => "Year",
            "month" => "Month",
            "day" => "Day",
            "chequeNumber" => "Cheque Number"
        ];
    }


    public function fillModel($customer, $userFullname, $billingCharge)
    {
        $this->username = $customer->username;
        $this->fullName = $userFullname;
        $balance =
            BillingModel::calculateOutstandingAmountOnBillingCharge(
                $billingCharge->id,
                $customer->personid
            );
        $this->amountPaid = $balance;
        $this->balance = $balance;
        $this->customerId = $customer->personid;
        $this->applicationPeriodId = $billingCharge->application_period_id;
        $this->billingChargeId = $billingCharge->id;

        $unformattedDate = date('Y-m-d');
        $yearSegment = substr($unformattedDate, 0, 4);
        $monthSegment = substr($unformattedDate, 5, 2);
        $daySegment = substr($unformattedDate, 8, 2);
        $this->yearPaid = $yearSegment;
        $this->monthPaid = $monthSegment;
        $this->dayPaid = $daySegment;
    }

    private function receiptNumberIdSegment()
    {
        $randId = rand(1, 9999);
        $numAsString = strval($randId);
        $fixedLengthString = null;
        $length = strlen($numAsString);
        if ($length == 1) {
            $fixedLengthString =  "000{$numAsString}";
        } elseif ($length == 2) {
            $fixedLengthString =  "00{$numAsString}";
        } elseif ($length == 3) {
            $fixedLengthString =  "0{$numAsString}";
        } elseif ($length == 4) {
            $fixedLengthString =  $numAsString;
        }
        return $fixedLengthString;
    }

    private function generateReceiptNumber()
    {
        $idSegment = $this->receiptNumberIdSegment();
        return "{$this->yearPaid}{$this->monthPaid}{$this->dayPaid}{$idSegment}";
    }


    private function generateReceipt(
        $studentRegistrationId,
        $customerId,
        $staffId
    ) {
        $receipt = new Receipt();
        $receipt->payment_method_id = $this->paymentMethodId;
        $receipt->customer_id = $customerId;
        $receipt->student_registration_id = $studentRegistrationId;
        $receipt->created_by = $staffId;
        $receipt->username = $this->username;
        $receipt->full_name = $this->fullName;
        $receipt->receipt_number = $this->generateReceiptNumber();

        $customer = UserModel::getUserById($customerId);
        $receipt->email = $customer->email;

        $receipt->date_paid =
            "{$this->yearPaid}-{$this->monthPaid}-{$this->dayPaid}";

        $receipt->timestamp = date("Y-m-d H:i:s");
        $receipt->cheque_number = $this->chequeNumber;
        $receipt->is_active = 0;
        $receipt->is_deleted = 1;
        if ($receipt->save() == true) {
            return $receipt;
        } else {
            return null;
        }
    }

    private function generateBilling(
        $receipt,
        $customerId,
        $staffId
    ) {
        $billing = new Billing();
        $billing->receipt_id = $receipt->id;
        $billing->payment_method_id = $receipt->payment_method_id;
        $billing->billing_charge_id = $this->billingChargeId;
        $billing->customer_id = $customerId;
        $billing->student_registration_id = $receipt->student_registration_id;

        $studentRegistration =
            StudentRegistration::find()
            ->where([
                "studentregistrationid" => $receipt->student_registration_id
            ])
            ->one();
        $billing->academic_offering_id = $studentRegistration->academicofferingid;
        $billing->application_period_id = $this->applicationPeriodId;
        $billing->created_by = $staffId;
        $billing->created_at = date("Y-m-d H:i:s");
        $billing->cost = $this->balance;
        $billing->amount_paid = $this->amountPaid;
        $billing->quantity = 1;
        $billing->is_active = 0;
        $billing->is_deleted = 1;
        return $billing;
    }


    public function processPaymentRequest(
        $staffId,
        $studentRegistrationId
    ) {
        if (
            $this->amountPaid < 0
            || $this->amountPaid > $this->balance
        ) {
            return new ErrorObject("Invalid payment amount");
        }

        $receipt =
            $this->generateReceipt(
                $studentRegistrationId,
                $this->customerId,
                $staffId
            );

        if ($receipt == null) {
            return new ErrorObject("Error ocurred generating receipt model");
        } else {
            $billing =
                $this->generateBilling(
                    $receipt,
                    $this->customerId,
                    $staffId
                );

            if ($billing->save() == false) {
                return new ErrorObject(
                    "Error ocurred generating application submission payment"
                        . " billing."
                );
            } else {
                return $receipt;
            }
        }
    }


    public function buildForGraduationPackageModification(
        $customer,
        $userFullname,
        $proposedBillingCharge,
        $maxPayableOnProposedBillingCharge
    ) {
        $this->username = $customer->username;
        $this->fullName = $userFullname;

        if ($maxPayableOnProposedBillingCharge <= 0) {
            $this->amountPaid = 0;
        } else {
            $this->amountPaid = $maxPayableOnProposedBillingCharge;
        }

        $this->balance = $proposedBillingCharge->cost;
        $this->customerId = $customer->personid;
        $this->applicationPeriodId = $proposedBillingCharge->application_period_id;
        $this->billingChargeId = $proposedBillingCharge->id;
        $unformattedDate = date('Y-m-d');
        $yearSegment = substr($unformattedDate, 0, 4);
        $monthSegment = substr($unformattedDate, 5, 2);
        $daySegment = substr($unformattedDate, 8, 2);
        $this->yearPaid = $yearSegment;
        $this->monthPaid = $monthSegment;
        $this->dayPaid = $daySegment;
    }


    private function validatePaymentAmount($packageModificationRequest)
    {
        if ($this->amountPaid < 0) {    // if payment is negative number
            return false;
        }

        $proposedPackageId = $this->billingChargeId;
        $proposedPackage =
            BillingCharge::find()
            ->where([
                "id" => $proposedPackageId,
                "is_active" => 1,
                "is_deleted" => 0
            ])
            ->one();

        $graduationBillings =
            Billing::find()
            ->innerJoin(
                'billing_charge',
                '`billing`.`billing_charge_id` = `billing_charge`.`id`'
            )
            ->innerJoin(
                'billing_type',
                '`billing_charge`.`billing_type_id` = `billing_type`.`id`'
            )->innerJoin(
                'billing_category',
                '`billing_type`.`billing_category_id` = `billing_category`.`id`'
            )
            ->where([
                "billing_charge.id" => $packageModificationRequest->current_package_id,
                "billing.student_registration_id" => $packageModificationRequest->student_registration_id,
                "billing.is_active" => 1,
                "billing.is_deleted" => 0,
                "billing_category.name" => "Graduation Services"
            ])
            ->all();

        if (empty($graduationBillings)) {   // if change request is based on expression of interest
            if ($this->amountPaid > $proposedPackage->cost) {
                return false;
            } else {
                return true;
            }
        } else {  // if change request was based on partial existing payment
            $totalOfCurrentPackagePaid = 0;
            foreach ($graduationBillings as $graduationBilling) {
                $totalOfCurrentPackagePaid += $graduationBilling->amount_paid;
            }

            $maxAmountPayable =
                $proposedPackage->cost - $totalOfCurrentPackagePaid;

            if (
                $this->amountPaidGreaterThanMaxAmountPayable($maxAmountPayable)
                == true
            ) {
                return false;
            }
            return true;
        }
    }

    private function amountPaidGreaterThanMaxAmountPayable($maxAmountPayable)
    {
        if ($maxAmountPayable <= 0 && $this->amountPaid > 0) {    // if payment is credit
            return true;
        }

        if ($maxAmountPayable > 0 && $this->amountPaid > $maxAmountPayable) {    // if amount is overpayment
            return true;
        }
        return false;
    }


    private function processPaymentForCredit(
        $svgccPersonnelId,
        $packageModificationRequest
    ) {
        $proposedPackage =
            BillingCharge::find()
            ->where([
                "id" =>  $packageModificationRequest->proposed_package_id,
                "is_active" => 1,
                "is_deleted" => 0
            ])
            ->one();

        $graduationBillings =
            Billing::find()
            ->innerJoin(
                'billing_charge',
                '`billing`.`billing_charge_id` = `billing_charge`.`id`'
            )
            ->innerJoin(
                'billing_type',
                '`billing_charge`.`billing_type_id` = `billing_type`.`id`'
            )->innerJoin(
                'billing_category',
                '`billing_type`.`billing_category_id` = `billing_category`.`id`'
            )
            ->where([
                "billing_charge.id" => $packageModificationRequest->current_package_id,
                "billing.student_registration_id" => $packageModificationRequest->student_registration_id,
                "billing.is_active" => 1,
                "billing.is_deleted" => 0,
                "billing_category.name" => "Graduation Services"
            ])
            ->all();

        $totalOfCurrentPackagePaid = 0;
        foreach ($graduationBillings as $graduationBilling) {
            $totalOfCurrentPackagePaid += $graduationBilling->amount_paid;
        }

        $paymentDelta = $proposedPackage->cost - $totalOfCurrentPackagePaid;
        if ($paymentDelta >= 0) {
            return null;
        } else {
            $credit = new Credit();
            $credit->student_registration_id = $packageModificationRequest->student_registration_id;
            $credit->person_id = $packageModificationRequest->person_id;
            $credit->amount = abs($paymentDelta);
            $credit->date_applied = date("Y-m-d H:i:s");;
            $credit->applied_by = $svgccPersonnelId;

            $credit->details =
                "Credit generated as a result of the total payments on a"
                . " graduation package exceeding the cost of most recently selected"
                . " package";

            $credit->credit_type_id = 1;
            $credit->was_refunded = null;
            $credit->date_refunded = null;
            $credit->refunded_by = null;
            $credit->is_deleted = 0;
            if ($credit->validate() == true) {
                return $credit;
            } else {
                return new ErrorObject("Error ocurred adding credit");
            }
        }
    }


    public function processGraduationPackageChangeRequest(
        $svgccPersonnelId,
        $packageModificationRequest
    ) {
        if ($this->validatePaymentAmount($packageModificationRequest) == false) {
            return new ErrorObject("Invalid payment amount entered");
        }

        $credit =
            $this->processPaymentForCredit(
                $svgccPersonnelId,
                $packageModificationRequest
            );

        if ($credit instanceof ErrorObject) {
            return $credit;
        } elseif ($credit instanceof Credit) {
            $credit->save();
        }

        $receipt =
            $this->generateReceipt(
                $packageModificationRequest->student_registration_id,
                $this->customerId,
                $svgccPersonnelId
            );

        if ($receipt == null) {
            return new ErrorObject("Error ocurred generating receipt model");
        }

        $packageModificationRequest->date_paid = date("Y-m-d H:i:s");
        $packageModificationRequest->graduation_package_change_request_status_id = 4;
        $packageModificationRequest->paid_by = $svgccPersonnelId;
        $packageModificationRequest->is_active = 0;
        $packageModificationRequest->save();

        $billing =
            $this->generateBilling(
                $receipt,
                $this->customerId,
                $svgccPersonnelId
            );

        if ($billing->save() == false) {
            return new ErrorObject(
                "Error ocurred generating application submission payment"
                    . " billing."
            );
        } else {
            return $receipt;
        }
    }

    private function generateGraduationSurvey($staffId, $studentRegistrationId)
    {
        $graduationSurvey = new GraduationSurvey();
        $graduationSurvey->student_registration_id = $studentRegistrationId;
        $graduationSurvey->creator_id = $staffId;
        $graduationSurvey->respondent_id = $staffId;
        $graduationSurvey->email = null;
        $graduationSurvey->send_date = null;
        $graduationSurvey->respond_date = date("Y-m-d H:i:s");
        $graduationSurvey->publish_count = 0;
        $graduationSurvey->will_participate = 1;
        $graduationSurvey->billing_charge_id = $this->billingChargeId;
        $graduationSurvey->is_deleted = 1;
        return $graduationSurvey;
    }


    public function processGraduationSurveyGenerationAndPaymentRequest($staffId, $studentRegistrationId)
    {
        if (
            $this->amountPaid < 0
            || $this->amountPaid > $this->balance
        ) {
            return new ErrorObject("Invalid payment amount");
        }

        $graduationSurvey = $this->generateGraduationSurvey($staffId, $studentRegistrationId);
        if ($graduationSurvey->save() == false) {
            return new ErrorObject("Error ocurred generating graduation survey.");
        }

        $receipt =
            $this->generateReceipt(
                $studentRegistrationId,
                $this->customerId,
                $staffId
            );

        if ($receipt == null) {
            return new ErrorObject("Error ocurred generating receipt model.");
        }

        $billing =
            $this->generateBilling(
                $receipt,
                $this->customerId,
                $staffId
            );

        if ($billing->save() == false) {
            return new ErrorObject("Error ocurred generating application submission payment billing.");
        }
        return $receipt;
    }
}
