<?php

namespace app\subcomponents\bursary\controllers;

use app\subcomponents\bursary\views\viewmodels\registryandlibraryservices\RegistryAndLibraryFeesViewModel;
use frontend\core\domain\exceptions\ApplicationException;
use frontend\core\ports\driver\registryandlibraryservices\ConfirmPaymentRequest;
use frontend\core\ports\driver\registryandlibraryservices\GenerateReceiptFileRequest;
use frontend\core\ports\driver\registryandlibraryservices\PublishReceiptRequest;
use frontend\core\ports\driver\registryandlibraryservices\PreviewPaymentRequest;
use frontend\core\ports\driver\registryandlibraryservices\ViewRegistryAndLibraryFeesRequest;
use frontend\core\services\ApplicationPeriodRepositoryImp;
use frontend\core\services\BillingChargeRepositoryImp;
use frontend\core\services\EmployeeRepositoryImp;
use frontend\core\services\ForGeneratingReceiptFileImp;
use frontend\core\services\ForPublishingReceiptImp;
use frontend\core\services\PaymentMethodRepositoryImp;
use frontend\core\services\ReceiptRepositoryImp;
use frontend\core\services\StudentRepositoryImp;
use frontend\core\services\StudentRegistrationRepositoryImp;
use frontend\core\services\UserRepositoryImp;
use frontend\core\usecases\ViewRegistryAndLibraryFeesUseCase;
use frontend\core\usecases\PreviewStudentServiceFeePaymentUseCase;
use frontend\core\usecases\ConfirmStudentServiceFeePaymentUseCase;
use frontend\core\usecases\GenerateReceiptFileUseCase;
use frontend\core\usecases\PublishReceiptUseCase;
use frontend\adapters\driven\real\ActiveRecordApplicationPeriodStorage;
use frontend\adapters\driven\real\ActiveRecordBillingChargeStorage;
use frontend\adapters\driven\real\ActiveRecordEmployeeStorage;
use frontend\adapters\driven\real\ActiveRecordPaymentMethodStorage;
use frontend\adapters\driven\real\ActiveRecordReceiptStorage;
use frontend\adapters\driven\real\ActiveRecordStudentStorage;
use frontend\adapters\driven\real\ActiveRecordStudentRegistrationStorage;
use frontend\adapters\driven\real\ActiveRecordUserStorage;
use frontend\adapters\driven\real\PdfFileGenerator;
use frontend\adapters\driven\real\SwiftMailerReceiptEmailPublisher;

use yii\base\Model;
use yii\web\Controller;
use Yii;


class StudentServicePaymentsController extends Controller
{
    /**
     * Process GET request to generate form including all possible library and
     * registry services
     *
     * @param int $studentRegistrationId
     * @return View
     */
    public function actionRegistryAndLibraryFees($studentRegistrationId)
    {
        $personnel = Yii::$app->user->identity;

        $studentRegistrationRepositoryImp =
            new StudentRegistrationRepositoryImp(
                new ActiveRecordStudentRegistrationStorage()
            );

        $userRepositoryImp =
            new UserRepositoryImp(new ActiveRecordUserStorage());

        $studentRepositoryImp =
            new StudentRepositoryImp(new ActiveRecordStudentStorage());

        $applicationPeriodRepositoryImp =
            new ApplicationPeriodRepositoryImp(
                new ActiveRecordApplicationPeriodStorage()
            );

        $paymentMethodRepositoryImp =
            new PaymentMethodRepositoryImp(
                new ActiveRecordPaymentMethodStorage()
            );

        $billingChargeRepositoryImp =
            new BillingChargeRepositoryImp(
                new ActiveRecordBillingChargeStorage()
            );

        $request =
            new ViewRegistryAndLibraryFeesRequest(
                $studentRegistrationId,
                $personnel->personid
            );

        $useCase =
            new ViewRegistryAndLibraryFeesUseCase(
                $studentRegistrationRepositoryImp,
                $applicationPeriodRepositoryImp,
                $userRepositoryImp,
                $studentRepositoryImp,
                $billingChargeRepositoryImp,
                $paymentMethodRepositoryImp
            );

        try {
            $response = $useCase->generateFeesForm($request);
        } catch (ApplicationException $ex) {
            $studentRegistration =
                $studentRegistrationRepositoryImp->getStudentRegistrationById(
                    $studentRegistrationId
                );

            $customer =
                $userRepositoryImp->getUserById($studentRegistration->personid);

            Yii::$app->getSession()->setFlash("warning", $ex->message);

            return $this->redirect([
                "student-payments/payment-options",
                "username" => $customer->username
            ]);
        }

        return $this->render(
            "registry-and-library-fees",
            ["serviceFeeCatalog" => $response->model]
        );
    }


    /**
     * Process POST request to generate preview of service fee payment showing
     * details of prospective receipt, excluding the 'receipt number' which
     * only is available after records are created in database
     *
     * @param int $studentRegistrationId
     * @return View
     */
    public function actionPreviewPayment($studentRegistrationId)
    {
        $personnel = Yii::$app->user->identity;
        $postData = Yii::$app->request->post();

        $studentRegistrationRepositoryImp =
            new StudentRegistrationRepositoryImp(
                new ActiveRecordStudentRegistrationStorage()
            );

        $userRepositoryImp =
            new UserRepositoryImp(new ActiveRecordUserStorage());

        $studentRepositoryImp =
            new StudentRepositoryImp(new ActiveRecordStudentStorage());

        $applicationPeriodRepositoryImp =
            new ApplicationPeriodRepositoryImp(
                new ActiveRecordApplicationPeriodStorage()
            );

        $paymentMethodRepositoryImp =
            new PaymentMethodRepositoryImp(
                new ActiveRecordPaymentMethodStorage()
            );

        $billingChargeRepositoryImp =
            new BillingChargeRepositoryImp(
                new ActiveRecordBillingChargeStorage()
            );

        $registryAndLibraryFeesViewModel =
            RegistryAndLibraryFeesViewModel::create(
                $studentRegistrationId,
                $studentRegistrationRepositoryImp,
                $applicationPeriodRepositoryImp,
                $userRepositoryImp,
                $studentRepositoryImp,
                $billingChargeRepositoryImp,
                $paymentMethodRepositoryImp,
                $personnel->personid
            );

        if (
            $registryAndLibraryFeesViewModel->receiptForm->load($postData)
            == true
            && Model::loadMultiple(
                $registryAndLibraryFeesViewModel->billingForms,
                $postData
            )
            == true
        ) {
            $employeeRepositoryImp =
                new EmployeeRepositoryImp(
                    new ActiveRecordEmployeeStorage()
                );

            $useCase =
                new PreviewStudentServiceFeePaymentUseCase(
                    $employeeRepositoryImp
                );

            $request =
                new PreviewPaymentRequest(
                    $personnel->personid,
                    $registryAndLibraryFeesViewModel
                );

            try {
                $response = $useCase->generateReceiptPreview($request);
            } catch (ApplicationException $ex) {
                Yii::$app->getSession()->setFlash("warning", $ex->message);

                return $this->render(
                    "registry-and-library-fees",
                    ["serviceFeeCatalog" => $registryAndLibraryFeesViewModel]
                );
            }

            return $this->render(
                "receipt-preview",
                [
                    "paymentPreview" => $response->model,
                    "serviceFeeCatalog" => $registryAndLibraryFeesViewModel
                ]
            );
        }

        Yii::$app->getSession()->setFlash(
            "warning",
            "Error occurred validating payment"
        );

        return $this->render(
            "registry-and-library-fees",
            ["serviceFeeCatalog" => $registryAndLibraryFeesViewModel]
        );
    }



    public function actionConfirmPayment($studentRegistrationId)
    {
        $postData = Yii::$app->request->post();
        $personnel = Yii::$app->user->identity;

        $studentRegistrationRepositoryImp =
            new StudentRegistrationRepositoryImp(
                new ActiveRecordStudentRegistrationStorage()
            );

        $userRepositoryImp =
            new UserRepositoryImp(new ActiveRecordUserStorage());

        $studentRepositoryImp =
            new StudentRepositoryImp(new ActiveRecordStudentStorage());

        $applicationPeriodRepositoryImp =
            new ApplicationPeriodRepositoryImp(
                new ActiveRecordApplicationPeriodStorage()
            );

        $paymentMethodRepositoryImp =
            new PaymentMethodRepositoryImp(
                new ActiveRecordPaymentMethodStorage()
            );

        $billingChargeRepositoryImp =
            new BillingChargeRepositoryImp(
                new ActiveRecordBillingChargeStorage()
            );

        $receiptRepositoryImp =
            new ReceiptRepositoryImp(new ActiveRecordReceiptStorage());

        $employeeRepositoryImp =
            new EmployeeRepositoryImp(new ActiveRecordEmployeeStorage());

        $registryAndLibraryFeesViewModel =
            RegistryAndLibraryFeesViewModel::create(
                $studentRegistrationId,
                $studentRegistrationRepositoryImp,
                $applicationPeriodRepositoryImp,
                $userRepositoryImp,
                $studentRepositoryImp,
                $billingChargeRepositoryImp,
                $paymentMethodRepositoryImp,
                $personnel->personid
            );

        if (
            $registryAndLibraryFeesViewModel->receiptForm->load($postData)
            == true
            && Model::loadMultiple(
                $registryAndLibraryFeesViewModel->billingForms,
                $postData
            )
            == true
        ) {
            $createRecordsUseCase =
                new ConfirmStudentServiceFeePaymentUseCase(
                    $receiptRepositoryImp,
                    $billingChargeRepositoryImp,
                    $employeeRepositoryImp
                );

            $createRecordsRequest =
                new ConfirmPaymentRequest(
                    $personnel->personid,
                    $registryAndLibraryFeesViewModel
                );

            try {
                $createRecordsResponse =
                    $createRecordsUseCase->confirmPayment($createRecordsRequest);

                $receiptFilesDirectory =
                    Yii::getAlias("@frontend")
                    . "/subcomponents/bursary/files/receipts/";

                $generateReceiptFileUseCase =
                    new GenerateReceiptFileUseCase(
                        new ForGeneratingReceiptFileImp(
                            new PdfFileGenerator($this, $receiptFilesDirectory)
                        ),
                        $receiptRepositoryImp
                    );

                $generateReceiptRequest =
                    new GenerateReceiptFileRequest($createRecordsResponse->receiptId);

                $generateReceiptFileResponse =
                    $generateReceiptFileUseCase->generateReceiptFile(
                        $generateReceiptRequest
                    );

                $publishReceiptUseCase =
                    new PublishReceiptUseCase(
                        new ForPublishingReceiptImp(
                            new SwiftMailerReceiptEmailPublisher(
                                $receiptFilesDirectory
                            )
                        ),
                        $receiptRepositoryImp
                    );

                $publishReceiptRequest =
                    new PublishReceiptRequest(
                        $generateReceiptFileResponse->receiptId
                    );

                $publishReceiptUseCase =
                    $publishReceiptUseCase->publishReceipt(
                        $publishReceiptRequest
                    );
            } catch (ApplicationException $ex) {
                Yii::$app->getSession()->setFlash("warning", $ex->message);

                return $this->render(
                    "registry-and-library-fees",
                    ["serviceFeeCatalog" => $registryAndLibraryFeesViewModel]
                );
            }

            return $this->redirect([
                "payments/view-receipt",
                "id" => $generateReceiptFileResponse->receiptId,
                "username" => $generateReceiptFileResponse->customerUsername
            ]);
        }
        Yii::$app->getSession()->setFlash(
            "warning",
            "Error occurred validating payment"
        );

        return $this->render(
            "registry-and-library-fees",
            ["serviceFeeCatalog" => $registryAndLibraryFeesViewModel]
        );
    }


    public function actionGraduationPackage($studentRegistrationId)
    {
        $personnel = Yii::$app->user->identity;

        return $this->render(
            "graduation-package",
            []
        );
    }
}
