<?php

namespace app\subcomponents\bursary\controllers;

use Yii;
use app\subcomponents\bursary\models\ApplyExemptionViewModel;
use common\models\AcademicOfferingModel;
use common\models\ApplicantModel;
use common\models\BatchStudentFeePaymentForm;
use common\models\BillingChargeExemption;
use common\models\BillingChargeModel;
use common\models\BillingTypeModel;
use common\models\BillingModel;
use common\models\Credit;
use common\models\ErrorObject;
use common\models\PaymentMethodModel;
use common\models\ReceiptModel;
use common\models\SingleStudentFeePaymentForm;
use common\models\StudentModel;
use common\models\StudentRegistrationModel;
use common\models\UserModel;
use yii\base\Model;
use yii\data\ArrayDataProvider;
use yii\helpers\ArrayHelper;

class StudentPaymentsController extends \yii\web\Controller
{
    public function actionScheduledFeeReport($username, $studentRegistrationId)
    {
        $user = Yii::$app->user->identity;

        $studentRegistration =
            StudentRegistrationModel::getStudentRegistrationByID(
                $studentRegistrationId
            );

        $application =
            StudentRegistrationModel::getApplication($studentRegistration);

        $customer = UserModel::getUserByUsername($username);
        $userFullname = UserModel::getUserFullname($customer);
        $applicant = ApplicantModel::getApplicantByPersonid($customer->personid);

        $applicationPeriodId =
            ApplicantModel::getApplicantApplicationPeriodID($applicant);

        $batchStudentFeePaymentForm = new BatchStudentFeePaymentForm();

        $batchStudentFeePaymentForm->fillModel(
            $customer,
            $user->personid,
            $userFullname,
            $applicationPeriodId
        );

        $batchStudentFeePaymentBillingForms =
            $batchStudentFeePaymentForm->generateDefaultBillingFormsForStudent(
                $studentRegistration
            );

        if (empty($batchStudentFeePaymentBillingForms)) {
            $outstandingFeesExist = false;
        } else {
            $outstandingFeesExist = true;
        }

        $paymentMethods =
            ArrayHelper::map(
                PaymentMethodModel::getNonWaiverPaymentMethods(),
                "paymentmethodid",
                "name"
            );

        $programme =
            AcademicOfferingModel::getProgrammeNameByStudentRegistrationId(
                $studentRegistrationId
            );

        $feesDataProvider =
            new ArrayDataProvider(
                [
                    "allModels" =>
                    StudentModel::prepareFeePaymentReportByRegistration(
                        $studentRegistration
                    ),
                    "pagination" => ["pageSize" => 100],
                    "sort" => [
                        "defaultOrder" => ["fee" => SORT_ASC],
                        "attributes" => ["fee"]
                    ]
                ]
            );

        $paymentSummary =
            ApplicantModel::calculateStudentEnrollmentFeesSummary(
                $application,
                $studentRegistration->currentlevel
            );
        $totalCost = $paymentSummary["totalCost"];
        $totalPaid = $paymentSummary["totalPaid"];
        $balanceDue = $paymentSummary["totalDue"];

        if ($postData = Yii::$app->request->post()) {
            if (
                $batchStudentFeePaymentForm->load($postData) == true
                && Model::loadMultiple(
                    $batchStudentFeePaymentBillingForms,
                    $postData
                )
                == true
            ) {
                $receipt =
                    $batchStudentFeePaymentForm->processEnrolledStudentPaymentRequest(
                        $batchStudentFeePaymentBillingForms,
                        $studentRegistrationId
                    );

                if ($receipt instanceof ErrorObject) {
                    Yii::$app->getSession()->setFlash(
                        "warning",
                        $receipt->getMessage()
                    );
                } else {
                    return $this->redirect([
                        "preview-receipt",
                        "receiptId" => $receipt->id,
                        "studentRegistrationId" => $studentRegistrationId
                    ]);
                }
            } else {
                Yii::$app->getSession()->setFlash(
                    "warning",
                    "Error occurred load payment"
                );
            }
        }

        return $this->render(
            "scheduled-fee-report",
            [
                "userFullname" => $userFullname,
                "username" => $customer->username,
                "dataProvider" => $feesDataProvider,
                "batchStudentFeePaymentForm" => $batchStudentFeePaymentForm,
                "paymentMethods" => $paymentMethods,
                "programme" => $programme,
                "studentRegistrationId" => $studentRegistrationId,

                "batchStudentFeePaymentBillingForms" =>
                $batchStudentFeePaymentBillingForms,

                "outstandingFeesExist" => $outstandingFeesExist,
                "totalCost" => $totalCost,
                "totalPaid" => $totalPaid,
                "balanceDue" => $balanceDue
            ]
        );
    }


    public function actionMakeFeePayment(
        $username,
        $billingChargeId,
        $studentRegistrationId
    ) {
        $user = Yii::$app->user->identity;

        $customer = UserModel::getUserByUsername($username);
        $userFullname = UserModel::getUserFullname($customer);

        $billingCharge =
            BillingChargeModel::getBillingChargeById($billingChargeId);

        $billingType =
            BillingTypeModel::getBillingTypeByID(
                $billingCharge->billing_type_id
            );

        $model = new SingleStudentFeePaymentForm();
        $model->fillModel($customer, $userFullname, $billingCharge);

        $paymentMethods =
            ArrayHelper::map(
                PaymentMethodModel::getActivePaymentMethods(),
                "paymentmethodid",
                "name"
            );

        if ($postData = Yii::$app->request->post()) {
            if ($model->load($postData) == true) {
                $receipt =
                    $model->processIndividualBillingPaymentRequest(
                        $user->personid,
                        $this
                    );

                if ($receipt instanceof ErrorObject) {
                    Yii::$app->getSession()->setFlash(
                        "warning",
                        $receipt->getMessage()
                    );
                } else {
                    return $this->redirect([
                        "preview-receipt",
                        "receiptId" => $receipt->id,
                        "studentRegistrationId" => $studentRegistrationId
                    ]);
                }
            } else {
                Yii::$app->getSession()->setFlash(
                    "warning",
                    "Error occurred loading payment"
                );
            }
        }

        return $this->render(
            "make-fee-payment",
            [
                "model" => $model,
                "userFullname" => $userFullname,
                "username" => $customer->username,
                "fee" => $billingType->name,
                "paymentMethods" => $paymentMethods,
                "studentRegistrationId" => $studentRegistrationId
            ]
        );
    }


    public function actionPreviewReceipt($receiptId, $studentRegistrationId)
    {
        $receipt = ReceiptModel::getReceiptById($receiptId);
        $billings = ReceiptModel::getBillings($receipt);
        $customer = UserModel::getUserById($receipt->customer_id);
        $applicantName = UserModel::getUserFullname($customer);
        $applicantId = $customer->username;
        $total = number_format(ReceiptModel::calculateReceiptTotal($receipt), 2);
        $user = UserModel::getUserById($receipt->created_by);
        $operator = ReceiptModel::getOperatorCode($user);

        return $this->render(
            "preview-receipt",
            [
                "receipt" => $receipt,
                "billings" => $billings,
                "total" => $total,
                "applicantName" => $applicantName,
                "applicantId" => $applicantId,
                "studentRegistrationId" => $studentRegistrationId,
                "operator" => $operator,
                "paymentMethod" => ReceiptModel::getPaymentMethod($receipt),
            ]
        );
    }


    public function actionRedoReceipt($receiptId, $studentRegistrationId)
    {
        $user = Yii::$app->user->identity;
        $receipt = ReceiptModel::getReceiptById($receiptId);
        $billings = ReceiptModel::getBillings($receipt);
        $customer = UserModel::getUserById($receipt->customer_id);
        $applicantId = $customer->username;

        if (ReceiptModel::deleteReceipt(
            $receipt,
            $billings,
            $user->personid
        ) == true) {
            return $this->redirect([
                "scheduled-fee-report",
                "username" => $applicantId,
                "studentRegistrationId" => $studentRegistrationId
            ]);
        } else {
            Yii::$app->getSession()->setFlash(
                'warning',
                'Error occurred deleting receipt.'
            );
        }
        return $this->redirect([
            "view-receipt",
            "id" => $receiptId,
            "username" => UserModel::getUserById($receipt->customer_id)->username,
        ]);
    }


    public function actionApproveAndPublishReceipt($receiptId)
    {
        $receipt = ReceiptModel::getReceiptById($receiptId);
        $billings = ReceiptModel::getBillings($receipt);
        $customer = UserModel::getUserById($receipt->customer_id);
        $applicantName = UserModel::getUserFullname($customer);
        $applicantId = $customer->username;

        ReceiptModel::publishReceipt(
            $this,
            $receipt,
            $billings,
            $applicantName,
            $applicantId
        );

        $receipt->publish_count += 1;
        if ($receipt->save() == true) {
            Yii::$app->getSession()->setFlash(
                "success",
                "Receipt published successfully."
            );
        } else {
            Yii::$app->getSession()->setFlash(
                "warning",
                "Error occurred publishing receipt."
            );
        }

        return $this->redirect([
            "payments/view-receipt",
            "id" => $receiptId,
            "username" => UserModel::getUserById($receipt->customer_id)->username
        ]);
    }


    public function actionPaymentOptions($username)
    {
        $customer = UserModel::findUserByUsername($username);

        $studentRegistrations =
            StudentModel::getStudentEnrollmentsSummary($customer->personid);

        $keys = array_keys($studentRegistrations);

        if (count($studentRegistrations) == 1) {
            return $this->redirect([
                "single-enrollment-payment-options",
                "username" => $username
            ]);
        } elseif (count($studentRegistrations) > 1) {
            return $this->redirect([
                "multiple-enrollments-payment-options",
                "username" => $username
            ]);
        }
    }

    public function actionSingleEnrollmentPaymentOptions($username)
    {
        $customer = UserModel::findUserByUsername($username);
        $userFullname = UserModel::getUserFullname($customer);

        $studentRegistrations =
            StudentModel::getStudentEnrollmentsSummary($customer->personid);

        $keys = array_keys($studentRegistrations);
        $studentRegistrationId = $keys[0];
        $programme = $studentRegistrations[$studentRegistrationId];

        return $this->render(
            "single-enrollment-payment-options",
            [
                "title" => "{$programme} Enrollment Payment Options",
                "username" => $username,
                "userFullname" => $userFullname,
                "studentRegistrationId" => $keys[0],
                "programme" => $programme
            ]
        );
    }

    public function actionMultipleEnrollmentsPaymentOptions($username)
    {
        $customer = UserModel::findUserByUsername($username);

        $userFullname = UserModel::getUserFullname($customer);

        $studentRegistrations =
            StudentModel::getStudentEnrollmentsSummary($customer->personid);

        return $this->render(
            "multiple-enrollments-payment-options",
            [
                "title" => "Payment Options",
                "username" => $username,
                "userFullname" => $userFullname,
                "studentRegistrations" => $studentRegistrations
            ]
        );
    }


    public function actionApplyExemption(
        $username,
        $billingChargeId,
        $studentRegistrationId
    ) {
        $svgccPersonnelId = Yii::$app->user->identity->personid;
        $viewModel =
            ApplyExemptionViewModel::create(
                $username,
                $studentRegistrationId,
                $billingChargeId,
                $svgccPersonnelId
            );

        if ($postData = Yii::$app->request->post()) {
            if ($viewModel->exemptionForm->load($postData) == true) {
                $credit = null;
                $billingChargeExemption =
                    $viewModel->exemptionForm->createExemption();

                if ($billingChargeExemption instanceof ErrorObject) {
                    Yii::$app->getSession()->setFlash(
                        "warning",
                        $billingChargeExemption->getMessage()
                    );
                }

                $studentRegistration =
                    StudentRegistrationModel::getStudentRegistrationByID(
                        $studentRegistrationId
                    );

                if (!empty(BillingModel::getCustomerFeePayments(
                    $billingChargeId,
                    $studentRegistration->personid
                ))) {
                    $credit =
                        $viewModel->exemptionForm->createCredit(
                            $studentRegistration->personid
                        );
                    if ($credit instanceof ErrorObject) {
                        Yii::$app->getSession()->setFlash(
                            "warning",
                            $credit->getMessage()
                        );
                    }
                }

                if ($credit instanceof Credit) {
                    $creditSaved = $credit->save();
                    $billingChargeExemption->credit_id = $credit->id;
                    if (
                        $creditSaved == true
                        && $billingChargeExemption->save() == true
                    ) {
                        Yii::$app->getSession()->setFlash(
                            "success",
                            "Exemption successful."
                        );
                        return $this->redirect([
                            "scheduled-fee-report",
                            "username" => $username,
                            "studentRegistrationId" => $studentRegistrationId
                        ]);
                    }
                } else {
                    if ($billingChargeExemption->save() == true) {
                        Yii::$app->getSession()->setFlash(
                            "success",
                            "Exemption successful."
                        );

                        return $this->redirect([
                            "scheduled-fee-report",
                            "username" => $username,
                            "studentRegistrationId" => $studentRegistrationId
                        ]);
                    }
                }
            } else {
                Yii::$app->getSession()->setFlash(
                    "warning",
                    "Error occurred load payment"
                );
            }
        }

        return $this->render("apply-exemption", ["viewModel" => $viewModel]);
    }


    public function actionUndoExemption(
        $username,
        $billingChargeId,
        $studentRegistrationId
    ) {
        $billingChargeExemption =
            BillingChargeExemption::find()
            ->where([
                "billing_charge_id" => $billingChargeId,
                "student_registration_id" => $studentRegistrationId,
                "is_deleted" => 0
            ])
            ->one();

        $applicableCredit =
            Credit::find()
            ->where([
                "id" => $billingChargeExemption->credit_id,
                "is_deleted" => 0
            ])
            ->one();

        if ($billingChargeExemption == true && $applicableCredit == true) {
            $billingChargeExemption->is_deleted = 1;
            $applicableCredit->is_deleted = 1;
            if (
                $billingChargeExemption->validate() == true
                && $applicableCredit->validate() == true
            ) {
                $billingChargeExemption->save();
                $applicableCredit->save();
            }
        } elseif ($billingChargeExemption == true && $applicableCredit == false) {
            $billingChargeExemption->is_deleted = 1;
            if ($billingChargeExemption->validate() == true) {
                $billingChargeExemption->save();
            }
        }

        return $this->redirect([
            "scheduled-fee-report",
            "username" => $username,
            "studentRegistrationId" => $studentRegistrationId
        ]);
    }
}
