<?php

namespace app\subcomponents\admissions\models;

use Yii;
use common\models\AcademicStatus;
use common\models\ApplicationModel;
use common\models\OfferModel;
use common\models\StudentRegistrationModel;
use common\models\StudentStatus;

class StudentRegistration
{

    public $studentRegistrationId;
    public $programme;
    public $level;
    public $studentStatus;
    public $academicStatus;
    public $semesterTranscripts;
    public $cumulativeGpa;
    public $transcriptFields;

    public function __construct(
        $studentRegistrationId,
        $programme,
        $level,
        $studentStatus,
        $academicStatus,
        $semesterTranscripts,
        $cumulativeGpa,
        $transcriptFields
    ) {
        $this->studentRegistrationId = $studentRegistrationId;
        $this->programme = $programme;
        $this->level = $level;
        $this->studentStatus = $studentStatus;
        $this->academicStatus = $academicStatus;
        $this->semesterTranscripts = $semesterTranscripts;
        $this->cumulativeGpa = $cumulativeGpa;
        $this->transcriptFields = $transcriptFields;
    }


    /**
     * StudentRegistration factory method
     *
     * @param integer $studentRegistrationId
     * @return StudentRegistration
     */
    public static function create($studentRegistrationId)
    {
        $registration =
            StudentRegistrationModel::getStudentRegistrationByID(
                $studentRegistrationId
            );

        $studentStatus =
            StudentStatus::find()
            ->where([
                "studentstatusid" => $registration->studentstatusid
            ])
            ->one()
            ->name;

        $academicStatus =
            AcademicStatus::find()
            ->where([
                "academicstatusid" => $registration->academicstatusid
            ])
            ->one()
            ->name;

        $transcriptFields = self::getFormattedHtmlTableHeader();

        return new StudentRegistration(
            $studentRegistrationId,
            self::getProgramme($registration),
            $registration->currentlevel,
            $studentStatus,
            $academicStatus,
            self::buildTranscript($studentRegistrationId),
            self::getCumulativeGpa($studentRegistrationId),
            $transcriptFields
        );
    }


    private static function getFormattedHtmlTableHeader()
    {
        return "<tr>"
            . "<th>Course Code</th>"
            . "<th>Course Name</th>"
            . "<th>Credits Attempted</th>"
            . "<th>Credits Awarded</th>"
            . "<th>CW</th>"
            . "<th>Exam</th>"
            . "<th>Final</th>"
            . "<th>Course Status</th>"
            . "<th>Grade</th>"
            . "<th>Quality Points</th> "
            . "<th>Grade Points</th> "
            . "</tr>";
    }


    private static function getProgramme($registration)
    {
        $offer = OfferModel::getOfferById($registration->offerid);
        $application = OfferModel::getApplication($offer);
        return ApplicationModel::getFormattedProgrammeChoice($application);
    }


    private static function getCumulativeGpa($studentRegistrationId)
    {
        $cumulative_gpa = "Pending";
        $gradepoints_sum = 0;
        $credits_sum = 0;

        $db = Yii::$app->db;
        $records = $db->createCommand(
            "SELECT course_offering.passfailtypeid AS 'passfailtypeid',"
                . " course_type.name AS 'course_type',"
                . " course_offering.credits AS 'credits',"
                . " batch_students.coursestatusid AS 'course_status',"
                . " batch_students.qualitypoints AS 'qualitypoints'"
                . " FROM student_registration"
                . " JOIN batch_students"
                . " ON student_registration.studentregistrationid = batch_students.studentregistrationid"
                . " JOIN batch"
                . " ON batch_students.batchid = batch.batchid"
                . " JOIN course_offering"
                . " ON batch.courseofferingid = course_offering.courseofferingid"
                . " JOIN course_type"
                . " ON course_offering.coursetypeid = course_type.coursetypeid"
                . " WHERE batch_students.isactive = 1 AND batch_students.isdeleted = 0"
                . " AND batch_students.studentregistrationid = " . $studentRegistrationId
                . ";"
        )
            ->queryAll();

        $records_count = count($records);
        if ($records_count > 0) {
            for ($i = 0; $i < $records_count; $i++) {
                $grade_points = $records[$i]['credits'] * $records[$i]['qualitypoints'];

                if (
                    strcmp($records[$i]['course_status'], 'INC') != 0
                    && strcmp($records[$i]['course_status'], 'EX') != 0
                    && ($records[$i]['passfailtypeid'] == 1
                        || $records[$i]['passfailtypeid'] == 3
                    )
                ) {
                    $gradepoints_sum += $grade_points;
                    $credits_sum += $records[$i]["credits"];
                }
            }
            if ($gradepoints_sum != 0  && $credits_sum != 0)
                $cumulative_gpa = round(($gradepoints_sum / $credits_sum), 2);
        }
        return $cumulative_gpa;
    }


    private static function buildTranscript($studentRegistrationId)
    {
        $semesterTranscripts = array();

        $courseResults =
            Yii::$app->db->createCommand(
                "SELECT course_catalog.coursecode AS 'courseCode',"
                    . " course_catalog.name AS 'courseName',"
                    . " course_offering.credits AS 'creditsAttempted',"
                    . " batch_students.studentregistrationid As 'studentRegistrationId',"
                    . " batch_students.courseworktotal As 'cw',"
                    . " batch_students.examtotal As 'exam',"
                    . " batch_students.final As 'final',"
                    . " course_status.abbreviation As 'courseStatus',"
                    . " batch_students.grade AS 'grade',"
                    . " batch_students.qualitypoints AS 'qualityPoints',"
                    . " semester.semesterid AS 'semesterId',"
                    . " semester.title AS 'semesterTitle',"
                    . " academic_year.academicyearid AS 'academicYearId',"
                    . " academic_year.title AS 'academicYearTitle'"
                    . " FROM batch_students"
                    . " JOIN batch"
                    . " ON batch_students.batchid = batch.batchid"
                    . " JOIN course_offering"
                    . " ON batch.courseofferingid = course_offering.courseofferingid"
                    . " JOIN course_catalog"
                    . " ON course_offering.coursecatalogid = course_catalog.coursecatalogid"
                    . " JOIN semester"
                    . " ON course_offering.semesterid = semester.semesterid"
                    . " JOIN academic_year"
                    . " ON semester.academicyearid = academic_year.academicyearid"
                    . " JOIN course_status"
                    . " ON batch_students.coursestatusid = course_status.coursestatusid"
                    . " WHERE batch_students.studentregistrationid = {$studentRegistrationId}"
                    . " AND batch_students.isactive = 1"
                    . " AND batch_students.isdeleted = 0"
                    . " ORDER BY semester.semesterid DESC, course_catalog.coursecode ASC;"
            )
            ->queryAll();

        if (empty($courseResults) == true) {
            return $semesterTranscripts;
        }

        $uniqueSemesterIds = array();
        $previousSemesterTranscript = null;
        $semesterTranscript = null;
        foreach ($courseResults as $courseResult) {
            $semesterId = $courseResult["semesterId"];
            if (in_array($semesterId, $uniqueSemesterIds) == false) {
                array_push($uniqueSemesterIds, $semesterId);
                if ($previousSemesterTranscript != null) {
                    array_push($semesterTranscripts, $previousSemesterTranscript);
                }
                $semesterTranscript = SemesterTranscript::create($courseResult);
            }
            $semesterTranscript->addGrade($courseResult);
            $previousSemesterTranscript = $semesterTranscript;
        }

        array_push($semesterTranscripts, $semesterTranscript);
        return $semesterTranscripts;
    }
}
