<?php

namespace frontend\core\usecases;

use frontend\core\domain\exceptions\ApplicationException;
use frontend\core\domain\bursary\BillingEntity;
use frontend\core\domain\bursary\ReceiptEntity;
use frontend\core\domain\bursary\PaymentMethod;
use frontend\core\ports\driver\registryandlibraryservices\ConfirmPaymentResponse;
use frontend\core\services\ReceiptNumberGenerator;
use Yii;

/**
 * Use case to process registry and library fee transaction.
 */
class ConfirmStudentServiceFeePaymentUseCase
{
    public $receiptRepository;
    public $billingChargeRepository;
    public $employeeRepository;

    /**
     * Class constructor
     *
     * @param ReceiptRepository $receiptRepository
     * @param BillingChargeRepository $billingChargeRepository
     * @param EmployeeRepository $employeeRepository
     */
    public function __construct(
        $receiptRepository,
        $billingChargeRepository,
        $employeeRepository
    ) {
        $this->receiptRepository = $receiptRepository;
        $this->billingChargeRepository = $billingChargeRepository;
        $this->employeeRepository = $employeeRepository;
    }


    /**
     * Confirm payment request.
     * 
     * This involve the creation of the relevant Receipt and Billings 
     * database records.
     * 
     * The generation of the receipt file and subsequent publishing are handled
     * by separate use cases.
     *
     * @param ConfirmPaymentRequest $request
     * @return ConfirmPaymentResponse
     */
    public function confirmPayment($request)
    {
        try {
            $billingEntities = $this->createBillingEntities($request);
            $receiptEntity = $this->createReceiptEntity($request);
            foreach ($billingEntities as $billingEntity) {
                $receiptEntity->addBilling($billingEntity);
            }

            $receipt = $this->receiptRepository->add($receiptEntity);
        } catch (ApplicationException $ex) {
            throw new ApplicationException(500, $ex->message);
        }

        return ConfirmPaymentResponse::createSuccessResponse(
            $receipt->id,
            $request->registryAndLibraryFeesViewModel->username
        );
    }


    /**
     * Process BillingForm models submitted by use to determine the payment
     * method classification fo the transaction
     *
     * @param array $billingForms
     * @return PaymentMethod
     */
    private function getPaymentMethod($billingForms)
    {
        $uniquePaymentMethods = array();
        foreach ($billingForms as $billing) {
            if (in_array($billing->paymentMethod, $uniquePaymentMethods) == false) {
                array_push($uniquePaymentMethods, $billing->paymentMethod);
            }
        }

        if (count($uniquePaymentMethods) == 1) {
            return $billingForms[0]->paymentMethod;
        } elseif (count($uniquePaymentMethods) > 1) {
            return PaymentMethod::COMBINED;
        }
    }


    /**
     * Builds ReceiptEntity model
     * 
     * This is an aggregate that stores both the receipt data as well at its
     * associated billing data
     *
     * @param ConfirmPaymentRequest $request
     * @return ReceiptEntity
     */
    private function createReceiptEntity($request)
    {
        $registryAndLibraryFeesPayment = $request->registryAndLibraryFeesViewModel;
        $receiptForm = $request->registryAndLibraryFeesViewModel->receiptForm;
        $billings = $request->registryAndLibraryFeesViewModel->billingForms;
        $timestamp = date("Y-m-d H:i:s");
        $total = $request->registryAndLibraryFeesViewModel->getBillingFormsTotal();

        $personnelCode =
            $request->registryAndLibraryFeesViewModel->getPersonnelCode(
                $this->employeeRepository
            );

        return ReceiptEntity::defaultConstructor(
            $this->getPaymentMethod($billings),
            $registryAndLibraryFeesPayment->customerId,
            $registryAndLibraryFeesPayment->studentRegistrationId,
            $registryAndLibraryFeesPayment->personnelId,
            $registryAndLibraryFeesPayment->personnelId,
            $registryAndLibraryFeesPayment->username,
            $registryAndLibraryFeesPayment->userFullName,
            ReceiptNumberGenerator::create(),
            $registryAndLibraryFeesPayment->customerEmail,
            $receiptForm->notes,
            1, // $publishCount = 1 => set to be published once initially
            1, // $autoPublish = 1 => receipt will be published automatically
            $receiptForm->datePaid,
            $timestamp,
            $receiptForm->chequeNumber,
            $timestamp,
            $total,
            $personnelCode
        );
    }


    /**
     * Add billings to receipt aggregate
     *
     * @param ConfirmPaymentRequest $request
     * @return array
     */
    private function createBillingEntities($request)
    {
        $billingEntities = array();

        $feesPayment = $request->registryAndLibraryFeesViewModel;
        $billingForms = $request->registryAndLibraryFeesViewModel->billingForms;
        $timestamp = date("Y-m-d H:i:s");

        if (!empty($billingForms)) {
            foreach ($billingForms as $billingForm) {
                if ($billingForm->isActive == 1) {
                    $billingChargeName =
                        $this->billingChargeRepository->getBillingChargeFeeNameById(
                            $billingForm->billingChargeId
                        );

                    array_push(
                        $billingEntities,
                        BillingEntity::defaultConstructor(
                            $billingForm->paymentMethod,
                            $billingChargeName,
                            $billingForm->billingChargeId,
                            $feesPayment->customerId,
                            $feesPayment->studentRegistrationId,
                            $feesPayment->academicOfferingId,
                            $feesPayment->applicationPeriodId,
                            $feesPayment->personnelId,
                            $timestamp,
                            $feesPayment->personnelId,
                            $timestamp,
                            $billingForm->cost,     // $cost
                            $billingForm->cost,     // $amountPaid
                            $billingForm->quantity
                        )   // $cost == $amountPaid as partial payment not allowed for service fees
                    );
                }
            }
        }

        return $billingEntities;
    }
}
