<?php

namespace frontend\core\services;

use frontend\core\domain\bursary\ReceiptRepository;
use frontend\core\domain\exceptions\RecordCreationException;

/**
 * ReceiptRepository Implementation
 */
class ReceiptRepositoryImp implements ReceiptRepository
{
    private $receiptStorage;


    /**
     * Class constructor
     *
     * @param [type] $receiptStorage
     */
    public function __construct($receiptStorage)
    {
        $this->receiptStorage = $receiptStorage;
    }


    /**
     * Returns DTO receipt model
     *
     * @param in $id
     * @return ReceiptEntity
     * @throws Exception
     */
    public function find($id)
    {
        $receiptDto = $this->receiptStorage->findReceiptById($id);
        $billingDtos = $this->receiptStorage->findReceiptBillings($id);
        return ReceiptConverter::toEntityAggregate($receiptDto, $billingDtos);
    }


    /**
     * Persist Receipt and its associated Billing records
     * 
     *
     * @param ReceiptEntity $receiptEntity
     * @return Receipt
     */
    public function add($receiptEntity)
    {
        $billingDtos = array();
        $billingEntities = $receiptEntity->getBillings();
        if (empty($billingEntities)) {
            throw new RecordCreationException(
                "Error occurred processing billings"
            );
        }

        $receiptDto = ReceiptConverter::toDto($receiptEntity);

        foreach ($billingEntities as $billingEntity) {
            array_push($billingDtos, BillingConverter::toDto($billingEntity));
        }

        $receipt =
            $this->receiptStorage->addReceiptAndBillings(
                $receiptDto,
                $billingDtos
            );

        return $receipt;
    }


    /**
     * Finds existing version of Receipt, transfers data in updated more
     * and save updated model to database.
     * 
     *
     * @param ReceiptEntity $receipt
     * @return void
     */
    public function update($receipt)
    {
    }


    public function getBillings($receiptEntity)
    {
        return $this->receiptStorage->getBillings($receiptEntity->id);
    }
}
