<?php

namespace frontend\adapters\driven\real;

use kartik\mpdf\Pdf;
use Mpdf\Mpdf;
use frontend\core\ports\driven\FileGenerator;
use app\subcomponents\bursary\views\viewmodels\registryandlibraryservices\ReceiptFileBilling;
use app\subcomponents\bursary\views\viewmodels\registryandlibraryservices\ReceiptFileModel;

/*
 * Driven adapter for generating receipt documents/files
 */

class PdfFileGenerator implements FileGenerator
{
    private $controller;
    private $filePath;


    /**
     * Class constructor
     *
     * @param \yii\web\Controller $controller
     * @param string $filePath
     */
    public function __construct($controller, $filePath)
    {
        $this->controller = $controller;
        $this->filePath = $filePath;
    }

    /**
     * Generates file
     *
     * @param ReceiptEntity $receiptEntity
     * @return void
     */
    public function generate($receiptEntity)
    {
        $content = $this->prepareReceiptContent($this->controller, $receiptEntity);
        $receiptNumber = $receiptEntity->getReceiptNumber();
        $filename = "{$this->filePath}/$receiptNumber.pdf";

        $pdf = new Pdf(
            [
                'filename' => $filename,
                'mode' => Pdf::MODE_CORE,
                'format' => Pdf::FORMAT_A4,
                'orientation' => Pdf::ORIENT_PORTRAIT,
                'destination' => Pdf::DEST_FILE,
                'content' => $content,
                'cssFile' =>
                '@vendor/kartik-v/yii2-mpdf/src/assets/kv-mpdf-bootstrap.min.css',
                'cssInline' => '.kv-heading-1{font-size:18px}',
                'options' => ['title' => "Receipt {$filename}"],
                'methods' => ['SetFooter' => ['{PAGENO}'],]
            ]
        );
        return $pdf->render();
    }

    /**
     * Undocumented function
     *
     * @param \yii\web\Controller $controller
     * @param ReceiptEntity $receiptEntity
     * @return string
     * @throws InvalidParamException if the view file does not exist.
     */
    private function prepareReceiptContent($controller, $receiptEntity)
    {
        $receiptTemplate = $this->getReceiptTemplate($receiptEntity);
        $billingEntities = $receiptEntity->getBillings();
        $receiptFileBillings = array();
        foreach ($billingEntities as $billingEntity) {
            array_push(
                $receiptFileBillings,
                new ReceiptFileBilling(
                    $billingEntity->getBillingCharge(),
                    $billingEntity->getCost(),
                    $billingEntity->getQuantity(),
                    $billingEntity->getAmountPaid()
                )
            );
        }

        $receiptFileModel =
            new ReceiptFileModel(
                $receiptEntity->getReceiptNumber(),
                $receiptEntity->getPersonnelCode(),
                $receiptEntity->getPaymentMethod(),
                date_format(
                    new \DateTime(
                        $receiptEntity->getDatePaid()
                    ),
                    "F j, Y"
                ),
                $receiptEntity->getUsername(),
                $receiptEntity->getFullName(),
                number_format($receiptEntity->getTotal(), 2),
                $receiptFileBillings
            );

        return $controller->renderPartial(
            $receiptTemplate,
            ["model" => $receiptFileModel]
        );
    }


    /**
     * Returns view name based on void status of receiptEntity
     *
     * @param ReceiptEntity $receiptEntity
     * @return string
     */
    private function getReceiptTemplate($receiptEntity)
    {
        if (
            $receiptEntity->getIsActive() == 0
            && $receiptEntity->getIsDeleted() == 0
        ) {
            return "voided-receipt-template";
        } else {
            return "receipt-template";
        }
    }
}
