<?php

namespace frontend\adapters\driven\real;

use frontend\adapters\driven\real\persistence\Billing;
use frontend\adapters\driven\real\persistence\BillingType;
use frontend\adapters\driven\real\persistence\Employee;
use frontend\adapters\driven\real\persistence\PaymentMethod;
use frontend\adapters\driven\real\persistence\Receipt;
use frontend\core\domain\exceptions\ModelNotFoundException;
use frontend\core\domain\exceptions\RecordCreationException;
use frontend\core\ports\driven\ReceiptStorage;
use frontend\core\domain\bursary\PaymentMethod as PaymentMethods;
use frontend\core\ports\driven\BillingDto;
use frontend\core\ports\driven\ReceiptDto;


/*
 * Driven adapter for querying receipt and billing db models and building their 
 * corresponding DTOs
 */

class ActiveRecordReceiptStorage implements ReceiptStorage
{

    /**
     * Persists receipts and associated billings.
     *
     * @param ReceiptDto $receiptDto
     * @param BillingDto[] $billingDtos
     * @return Receipt
     */
    public function addReceiptAndBillings($receiptDto, $billingDtos)
    {
        $receipt = $this->createReceiptRecord($receiptDto);

        foreach ($billingDtos as $billingDto) {
            $billing = $this->createBillingRecord($billingDto, $receipt);
        }

        return $receipt;
    }


    /**
     * Returns billing models associated with receipt
     *
     * @param integer $receiptId
     * @return BillingDto[]
     * @throws ModelNotFoundException
     */
    public function findReceiptBillings($receiptId)
    {
        $billingRecords =
            Billing::find()
            ->where([
                "receipt_id" => $receiptId,
                "is_active" => 1,
                "is_deleted" => 0
            ])
            ->all();

        if ($billingRecords == false) {
            throw new ModelNotFoundException("No billings found.");
        }
        return $this->getBillingDtos($billingRecords);
    }


    /**
     * Finds and return receipt model
     *
     * @param integer $id
     * @return ReceiptDto
     * @throws ModelNotFoundException
     */
    public function findReceiptById($id)
    {
        $receiptRecord = Receipt::find()->where(["id" => $id])->one();

        if ($receiptRecord === null) {
            throw new ModelNotFoundException("Receipt not found.");
        }
        return $this->getReceiptDto($receiptRecord);
    }


    /**
     * Converts receipt database model
     *
     * @param ReceiptDto $receiptDto
     * @return Receipt
     * @throws RecordCreationException
     */
    private function createReceiptRecord($receiptDto)
    {
        $receipt = new Receipt();
        $paymentMethod = $this->getPaymentMethodByName($receiptDto->paymentMethod);
        $receipt->payment_method_id = $paymentMethod->paymentmethodid;
        $receipt->customer_id = $receiptDto->customerId;
        $receipt->student_registration_id = $receiptDto->studentRegistrationId;
        $receipt->created_by = $receiptDto->creatorId;
        $receipt->modified_by = $receiptDto->modifierId;
        $receipt->username = $receiptDto->username;
        $receipt->full_name = $receiptDto->fullName;
        $receipt->receipt_number = $receiptDto->receiptNumber;
        $receipt->email = $receiptDto->email;
        $receipt->notes = $receiptDto->notes;
        $receipt->publish_count = $receiptDto->publishCount;
        $receipt->auto_publish = $receiptDto->autoPublish;
        $receipt->date_paid = $receiptDto->datePaid;
        $receipt->timestamp = $receiptDto->createdTimestamp;
        $receipt->is_active = $receiptDto->isActive;
        $receipt->is_deleted = $receiptDto->isDeleted;
        $receipt->cheque_number = $receiptDto->chequeNumber;
        $receipt->modified_at = $receiptDto->lastModifiedTimestamp;
        if ($receipt->save() == false) {
            throw new RecordCreationException(
                "Error occurred generating receipt record"
            );
        }
        return $receipt;
    }


    /**
     * Builds billing database model
     *
     * @param BillingDto $billingDto
     * @param Receipt $receipt
     * @return Billing
     * @throws RecordCreationException
     */
    private function createBillingRecord($billingDto, $receipt)
    {
        $billing = new Billing();
        $billing->receipt_id = $receipt->id;
        $paymentMethod = $this->getPaymentMethodByName($billingDto->paymentMethod);
        $billing->payment_method_id = $paymentMethod->paymentmethodid;
        $billing->billing_charge_id = $billingDto->billingChargeId;
        $billing->customer_id = $billingDto->customerId;
        $billing->student_registration_id = $billingDto->studentRegistrationId;
        $billing->academic_offering_id = $billingDto->academicOfferingId;
        $billing->application_period_id = $billingDto->applicationPeriodId;
        $billing->created_by = $billingDto->creatorId;
        $billing->modified_by = $billingDto->modifierId;
        $billing->cost = $billingDto->cost;
        $billing->amount_paid = $billingDto->amountPaid;
        $billing->is_active = $billingDto->isActive;
        $billing->is_deleted = $billingDto->isDeleted;
        $billing->created_at = $billingDto->createdTimestamp;
        $billing->modified_at = $billingDto->lastModifiedTimestamp;
        $billing->quantity = $billingDto->quantity;
        if ($billing->save() == false) {
            throw new RecordCreationException(
                "Error occurred generating billing record"
            );
        }
        return $billing;
    }


    /**
     * Get PaymentMethod based on $name
     *
     * @param string $name
     * @return PaymentMethod
     * @throws ModelNotFoundException
     */
    private function getPaymentMethodByName($name)
    {
        $model = PaymentMethod::find()->where(["name" => $name])->one();
        if ($model === null) {
            throw new ModelNotFoundException("Payment method not found.");
        }
        return $model;
    }


    /**
     * Convert collection of Billing database models to BillingDtos
     *
     * @param Billing $billingRecords
     * @return BillingDto[]
     */
    private function getBillingDtos($billingRecords)
    {
        $billingDtos = array();
        if (!empty($billingRecords)) {
            foreach ($billingRecords as $billingRecord) {
                array_push(
                    $billingDtos,
                    new BillingDto(
                        $billingRecord->id,
                        $billingRecord->payment_method_id,
                        $this->getBillingCharge($billingRecord->billing_charge_id),
                        $billingRecord->billing_charge_id,
                        $billingRecord->customer_id,
                        $billingRecord->student_registration_id,
                        $billingRecord->academic_offering_id,
                        $billingRecord->application_period_id,
                        $billingRecord->created_by,
                        $billingRecord->created_at,
                        $billingRecord->modified_by,
                        $billingRecord->modified_at,
                        $billingRecord->cost,
                        $billingRecord->amount_paid,
                        $billingRecord->quantity,
                        $billingRecord->is_active,
                        $billingRecord->is_deleted
                    )
                );
            }
        }
        return $billingDtos;
    }


    /**
     * Returns the billing charge type
     *
     * @param integer $billingChargeId
     * @return string
     */
    private function getBillingCharge($billingChargeId)
    {
        $billingType =
            BillingType::find()
            ->innerJoin(
                'billing_charge',
                '`billing_type`.`id` = `billing_charge`.`billing_type_id`'
            )
            ->where(["billing_charge.id" => $billingChargeId])
            ->all();

        if ($billingType == true) {
            return $billingType[0]->name;
        }
        return null;
    }


    /**
     * Convert receipt database models to ReceiptDto
     *
     * @param Receipt $receiptRecord
     * @return ReceiptDto
     */
    private function getReceiptDto($receiptRecord)
    {
        return new ReceiptDto(
            $receiptRecord->id,
            $this->getPaymentMethodById($receiptRecord->payment_method_id),
            $receiptRecord->customer_id,
            $receiptRecord->student_registration_id,
            $receiptRecord->created_by,
            $receiptRecord->modified_by,
            $receiptRecord->username,
            $receiptRecord->full_name,
            $receiptRecord->receipt_number,
            $receiptRecord->email,
            $receiptRecord->notes,
            $receiptRecord->publish_count,
            $receiptRecord->auto_publish,
            $receiptRecord->date_paid,
            $receiptRecord->timestamp,
            $receiptRecord->is_active,
            $receiptRecord->is_deleted,
            $receiptRecord->cheque_number,
            $receiptRecord->modified_at,
            $this->getReceiptTotal($receiptRecord),
            $this->getOperatorCode($receiptRecord)
        );
    }


    /**
     * Return payment method by $id
     *
     * @param integer $id
     * @return PaymentMethod
     * @throws ModelNotFoundException
     */
    private function getPaymentMethodById($id)
    {
        $model = PaymentMethod::find()->where(["paymentmethodid" => $id])->one();
        if ($model === null) {
            throw new ModelNotFoundException("Payment method not found.");
        }
        return $model->name;
    }


    /**
     * Performs summation of a receipt's billings
     *
     * @param Receipt $receiptRecord
     * @return float
     */
    private function getReceiptTotal($receiptRecord)
    {
        $billingDtos = $this->findReceiptBillings($receiptRecord->id);
        $total = 0;
        foreach ($billingDtos as $billingDto) {
            $total += ($billingDto->amountPaid * $billingDto->quantity);
        }
        return $total;
    }


    /**
     * Generate employee receipt code
     *
     * @param Receipt $receiptRecord
     * @return Employee
     * @throws ModelNotFoundException
     */
    private function getOperatorCode($receiptRecord)
    {
        $employee =
            Employee::find()
            ->where(["personid" => $receiptRecord->created_by])
            ->one();
        if ($employee === null) {
            throw new ModelNotFoundException("Employee not found.");
        } else {
            $last = substr($employee->lastname, 0, 3);
            $first = substr($employee->firstname, 0, 3);
            $code = "{$last}{$first}";
            return strtoupper($code);
        }
    }
}
