<?php

namespace frontend\adapters\driven\real;

use frontend\adapters\driven\real\persistence\BillingCharge;
use frontend\adapters\driven\real\persistence\BillingType;
use frontend\core\domain\exceptions\ModelNotFoundException;
use frontend\core\ports\driven\BillingChargeStorage;

/**
 * Driven adapter for managing billing charge models
 */
class ActiveRecordBillingChargeStorage implements BillingChargeStorage
{
    /**
     * Returns all registry and library service fees for an application period
     *
     * @param ApplicationPeriod $applicationPeriod
     * @return BillingCharge
     * @throws ModelNotFoundException
     */
    public function getRegistryAndLibraryServiceBillingChargesForApplicationPeriod(
        $applicationPeriod
    ) {
        $charges =
            BillingCharge::find()
            ->innerJoin(
                'billing_type',
                '`billing_charge`.`billing_type_id` = `billing_type`.`id`'
            )
            ->innerJoin(
                'billing_category',
                '`billing_type`.`billing_category_id` = `billing_category`.`id`'
            )
            ->where([
                "billing_charge.application_period_id" => $applicationPeriod->applicationperiodid,
                "billing_charge.is_active" => 1,
                "billing_charge.is_deleted" => 0,
                "billing_category.name" => [
                    "Registry Services",
                    "Library Services"
                ]
            ])
            ->all();
        if ($charges == false) {
            throw new ModelNotFoundException(
                "No student services fees exist for the application period in question."
            );
        } else {
            return $charges;
        }
    }


    /**
     * Returns the name of the type a billing charge is associated with
     *
     * @param BillingCharge $billingCharge
     * @return string|null
     */
    public function getBillingChargeFeeName($billingCharge)
    {
        $billingType =
            BillingType::find()
            ->where(["id" => $billingCharge->billing_type_id])
            ->one();

        if ($billingType == true) {
            return $billingType->name;
        } else {
            return null;
        }
    }


    /**
     * Returns billing charge by $id
     *
     * @param integer $id
     * @return BillingCharge
     * @throws ModelNotFoundException
     */
    public function getBillingChargeById($id)
    {
        $model = BillingCharge::find()->where(["id" => $id])->one();
        if ($model === null) {
            throw new ModelNotFoundException("Billing charge not found.");
        }
        return $model;
    }


    /**
     * Returns the name of the type a billing charge is associated with
     *
     * @param integer $billingChargeId
     * @return string
     * @throws ModelNotFoundException
     */
    public function getBillingChargeFeeNameById($billingChargeId)
    {
        $billingCharge = $this->getBillingChargeById($billingChargeId);

        $billingType =
            BillingType::find()
            ->where(["id" => $billingCharge->billing_type_id])
            ->one();

        if ($billingType == true) {
            return $billingType->name;
        }
        return null;
    }
}
