<?php

namespace common\models;

class StudentModel
{
    public static function getStudentByPersonid($id)
    {
        return Student::find()->where(["personid" => $id])->one();
    }


    public static function getNameWithMiddleName($student)
    {
        return "{$student->title} "
            . "{$student->firstname} "
            . "{$student->middlename} "
            . "{$student->lastname}";
    }


    public static function getNameWithoutMiddleName($student)
    {
        return "{$student->title} {$student->firstname} {$student->lastname}";
    }


    public static function getStudentFullName($student)
    {
        if ($student == false) {
            return null;
        } elseif ($student == true && self::hasMiddleName($student) == true) {
            return self::getNameWithMiddleName($student);
        } elseif ($student == true && self::hasMiddleName($student) == false) {
            return self::getNameWithoutMiddleName($student);
        }
    }


    public static function hasMiddleName($student)
    {
        if ($student->middlename == true) {
            return true;
        }
        return false;
    }


    public static function getCurrentProgramme($student)
    {
        $activeRegistration =
            StudentRegistrationModel::getActiveStudentRegistrationByPersonID(
                $student->personid
            );

        $offer = OfferModel::getOfferById($activeRegistration->offerid);
        $currentApplication = OfferModel::getApplication($offer);
        return ApplicationModel::getFormattedProgrammeChoice($currentApplication);
    }

    public static function getActiveStudentByPersonid($id)
    {
        return Student::find()
            ->where(["personid" => $id, "isactive" => 1, "isdeleted" => 0])
            ->one();
    }


    public static function getStudentRegistrations($personid)
    {
        $registrations = array();

        $studentRegistrations =
            StudentRegistrationModel::getRegistrationsByPersonId($personid);

        foreach ($studentRegistrations as $registration) {
            $id = $registration->studentregistrationid;

            $academicOffering =
                AcademicOfferingModel::getAcademicOfferingByID(
                    $registration->academicofferingid
                );

            $programme =
                AcademicOfferingModel::getProgrammeName($academicOffering);

            $registrations[$id] = $programme;
        }

        return $registrations;
    }


    public static function prepareFeePaymentReportByRegistration(
        $studentRegistration
    ) {
        $data = array();

        $application =
            Application::find()
            ->innerJoin(
                'offer',
                '`application`.`applicationid` = `offer`.`applicationid`'
            )
            ->innerJoin(
                'student_registration',
                '`offer`.`offerid` = `student_registration`.`offerid`'
            )
            ->where([
                // "application.isactive" => 1,
                "application.isdeleted" => 0,
                "student_registration.studentregistrationid" => $studentRegistration->studentregistrationid
            ])
            ->one();

        $applicableBillingCharges =
            BillingChargeModel::getAllBillingChargesForApplication(
                $application
            );

        foreach ($applicableBillingCharges as $billingCharge) {
            $charge = array();
            $customer = UserModel::getUserById($studentRegistration->personid);
            $charge["studentRegistrationId"] = $studentRegistration->studentregistrationid;
            $charge["username"] = $customer->username;
            $charge["billingChargeId"] = $billingCharge->id;
            $charge["customerId"] = $studentRegistration->personid;

            $billingType =
                BillingTypeModel::getBillingTypeByID(
                    $billingCharge->billing_type_id
                );
            $charge["fee"] = $billingType->name;

            $charge["cost"] = number_format($billingCharge->cost, 2);

            $totalPaid =
                BillingModel::calculateTotalPaidOnBillingCharge(
                    $billingCharge->id,
                    $studentRegistration->personid
                );

            if (self::studentHasFullFeeExemption(
                $billingCharge,
                $studentRegistration
            ) == true) {
                $charge["totalPaid"] = "N/A";
            } else {
                $charge["totalPaid"] = number_format($totalPaid, 2);
            }

            $outstanding = $billingCharge->cost - $totalPaid;
            $charge["outstanding"] = number_format($outstanding, 2);

            $charge["status"] = self::getBillingChargePaymentStatus(
                $billingCharge,
                $studentRegistration
            );

            if (
                self::isPaymentAllowed($billingCharge, $studentRegistration)
                == true
            ) {
                $charge["paymentSubmissionAllowed"] = true;
            } else {
                $charge["paymentSubmissionAllowed"] = false;
            }

            $billingChargeExemption =
                BillingChargeExemption::find()
                ->where([
                    "billing_charge_id" => $billingCharge->id,
                    "student_registration_id" => $studentRegistration->studentregistrationid,
                    "is_deleted" => 0
                ])
                ->one();

            if ($billingChargeExemption == true) {
                $charge["exemption"] =
                    "<b>Exemption Amount</b> - $ {$billingChargeExemption->amount}<br/> <b>Exemption Details</b> - {$billingChargeExemption->notes}";
            } else {
                $charge["exemption"] = null;
            }


            $data[] = $charge;
        }
        return $data;
    }

    public static function studentHasFullFeeExemption(
        $billingCharge,
        $studentRegistration
    ) {
        $exemption =
            BillingChargeExemption::find()
            ->where([
                "billing_charge_id" => $billingCharge->id,
                "student_registration_id" => $studentRegistration->studentregistrationid,
                "is_deleted" => 0
            ])
            ->one();

        if ($exemption == true && $exemption->amount == $billingCharge->cost) {
            return true;
        }
        return false;
    }


    public static function studentHasPartialFeeExemption(
        $billingCharge,
        $studentRegistration
    ) {
        $exemption =
            BillingChargeExemption::find()
            ->where([
                "billing_charge_id" => $billingCharge->id,
                "student_registration_id" => $studentRegistration->studentregistrationid,
                "is_deleted" => 0
            ])
            ->one();

        if ($exemption == true && $exemption->amount < $billingCharge->cost) {
            return true;
        }
        return false;
    }


    public static function isPaymentAllowed($billingCharge, $studentRegistration)
    {
        $totalPaid =
            BillingModel::calculateTotalPaidOnBillingCharge(
                $billingCharge->id,
                $studentRegistration->personid
            );

        if (self::studentHasFullFeeExemption(
            $billingCharge,
            $studentRegistration
        ) == true) {
            return false;
        }

        if (
            ($totalPaid == 0
                && $billingCharge->payable_on_enrollment == 1)
            || ($totalPaid == 0
                && $billingCharge->payable_on_enrollment == 0
                && $studentRegistration->currentlevel == 2)
            || ($totalPaid > 0 && $totalPaid < $billingCharge->cost)
        ) {
            return true;
        } else {
            return false;
        }
    }

    public static function getBillingChargePaymentStatus(
        $billingCharge,
        $studentRegistration
    ) {
        $status = null;

        $billingChargeExemption =
            BillingChargeExemption::find()
            ->where([
                "billing_charge_id" => $billingCharge->id,
                "student_registration_id" => $studentRegistration->studentregistrationid,
                "is_deleted" => 0
            ])
            ->one();

        if (
            $billingChargeExemption == true
            && $billingChargeExemption->amount == $billingCharge->cost
        ) {
            $status =
                "<b>Full Fee Exemption</b><br/>"
                . " <b>Amount</b> - $ {$billingChargeExemption->amount}<br/>"
                . " <b>Details</b> - {$billingChargeExemption->notes}";
        } elseif (
            $billingChargeExemption == true
            && $billingChargeExemption->amount < $billingCharge->cost
        ) {
            $totalPaid =
                BillingModel::calculateTotalPaidOnBillingCharge(
                    $billingCharge->id,
                    $studentRegistration->personid
                );
            $formattedTotalPaid = number_format($totalPaid, 2);


            $outstanding =
                $billingCharge->cost - ($totalPaid + $billingChargeExemption->amount);
            $formattedOutstanding = number_format($outstanding, 2);

            $status = "<b>Partial Fee Exemption</b><br/>";

            $status .=
                " <b>Exemption Amount</b> - $ {$billingChargeExemption->amount}<br/>";

            $status .=
                " <b>Payment Amount</b> - $ {$formattedTotalPaid}<br/>";

            $status .=
                " <b>Balance</b> - $ {$formattedOutstanding}<br/>";

            $status .= " <b>Exemption Details</b> - {$billingChargeExemption->notes}";
        } else if ($billingChargeExemption == false) {
            $totalPaid =
                BillingModel::calculateTotalPaidOnBillingCharge(
                    $billingCharge->id,
                    $studentRegistration->personid
                );

            $outstanding = $billingCharge->cost - $totalPaid;

            if (
                ($totalPaid == 0
                    && $billingCharge->payable_on_enrollment == 1)
                || ($totalPaid == 0
                    && $billingCharge->payable_on_enrollment == 0
                    && $studentRegistration->currentlevel == 2)
            ) {
                $status = "Full Amount Outstanding";
            } elseif (
                $totalPaid == 0
                && $billingCharge->payable_on_enrollment == 0
                && $studentRegistration->currentlevel == 1
            ) {
                $status = "Due On Promotion to Year 2";
            } elseif ($totalPaid > 0 && $totalPaid < $billingCharge->cost) {
                $outstandingFormatted = number_format($outstanding, 2);;
                $status = "Outstanding = {$outstandingFormatted}";
            } elseif ($totalPaid == $billingCharge->cost) {
                $status = "Paid In Full";
            }
        }

        return $status;
    }


    public static function getStudentEnrollmentsSummary($personid)
    {
        $registrations = array();

        $studentRegistrations =
            StudentRegistrationModel::getRegistrationsByPersonId($personid);

        foreach ($studentRegistrations as $registration) {
            $id = $registration->studentregistrationid;

            $academicOffering =
                AcademicOfferingModel::getAcademicOfferingByID(
                    $registration->academicofferingid
                );

            $programme =
                AcademicOfferingModel::getProgrammeFullName($academicOffering);

            $registrations[$id] = $programme;
        }

        return $registrations;
    }
}
