<?php

namespace common\models;

use yii\base\Model;

class BatchStudentFeePaymentForm extends Model
{
    public $username;
    public $fullName;
    public $customerId;
    public $staffId;
    public $applicationPeriodId;
    public $datePaid;
    public $chequeNumber;
    public $payUsingCheque;

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [
                [
                    "username",
                    "fullName",
                    "datePaid",
                    "customerId",
                    "staffId",
                    "applicationPeriodId",
                    "payUsingCheque"
                ],
                "required"
            ],
            [["username", "fullName", "chequeNumber"], "string"],
            [
                [
                    "customerId",
                    "staffId",
                    "applicationPeriodId",
                    "payUsingCheque"
                ],
                "integer"
            ],
            [["datePaid"], "safe"]
        ];
    }


    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [

            "username" => "ApplicantID",
            "fullName" => "Full Name",
            "datePaid" => "Date of payment",
            "chequeNumber" => "Cheque Number",
            "payUsingCheque" => "Is any part of this payment made using cheque?"
        ];
    }


    public function fillModel(
        $customer,
        $staffId,
        $userFullName,
        $applicationPeriodId,
        $chequeNumber = NULL
    ) {
        $this->username = $customer->username;
        $this->fullName = $userFullName;
        $this->customerId = $customer->personid;
        $this->staffId = $staffId;
        $this->applicationPeriodId = $applicationPeriodId;
        $this->chequeNumber = $chequeNumber;
    }


    // public function generateDefaultBillingFormsForSuccessfulApplicant(
    //     $academicOffering
    // ) {
    //     $records = array();

    //     $applicableBillingCharges =
    //         BillingChargeModel::getOutstandingEnrollmentChargesByApplicationPeriodId(
    //             $academicOffering
    //         );

    //     foreach ($applicableBillingCharges as $billingCharge) {
    //         $record = new BatchStudentFeePaymentBillingForm();
    //         $record->fillModel($this->customerId, $billingCharge);
    //         $records[] = $record;
    //     }

    //     return $records;
    // }
    public function generateDefaultBillingFormsForSuccessfulApplication(
        $application
    ) {
        $records = array();

        $applicableBillingCharges =
            BillingChargeModel::getOutstandingEnrollmentChargesByApplication(
                $application
            );

        if (!empty($applicableBillingCharges)) {
            foreach ($applicableBillingCharges as $billingCharge) {
                $record = new BatchStudentFeePaymentBillingForm();
                $record->fillModel($this->customerId, $billingCharge);
                $records[] = $record;
            }
        }

        return $records;
    }


    private function getCustomerStudentRegistration(
        $personId,
        $applicationPeriodId
    ) {
        $registrations =  StudentRegistration::find()
            ->innerJoin(
                'academic_offering',
                '`student_registration`.`academicofferingid` = `academic_offering`.`academicofferingid`'
            )
            ->where([
                "academic_offering.applicationperiodid" => $applicationPeriodId,
                "student_registration.personid" => $personId,
                "student_registration.isdeleted" => 0
            ])
            ->all();
        if (!empty($registrations)) {
            return $registrations[0];
        } else {
            return false;
        }
    }

    private function padToFourCharacterString($numAsString)
    {
        $length = strlen($numAsString);
        if ($length == 1) {
            return "000{$numAsString}";
        } elseif ($length == 2) {
            return "00{$numAsString}";
        } elseif ($length == 3) {
            return "0{$numAsString}";
        } elseif ($length == 4) {
            return $numAsString;
        }
    }

    private function getLastReceiptId()
    {
        $receipts = Receipt::find()->orderBy("id DESC")->all();
        if (!empty($receipts)) {
            $id = $receipts[0]->id;
            $idAsFourCharacterString = strval($id % 10000);
            return $this->padToFourCharacterString($idAsFourCharacterString);
        } else {
            return "0000";
        }
    }

    private function generateReceiptNumber()
    {
        $unformattedDate = date('Y-m-d');
        $yearSegment = substr($unformattedDate, 0, 4);
        $monthSegment = substr($unformattedDate, 5, 2);
        $daySegment = substr($unformattedDate, 8, 2);
        $idSegment = $this->getLastReceiptId();

        return "{$yearSegment}{$monthSegment}{$daySegment}{$idSegment}";
    }


    public function generateBillings($receipt, $billings)
    {
        $savedBillings = array();
        if ($billings == false) {
            return false;
        } else {
            foreach ($billings as $billing) {
                if ($billing->validateModel() == true) {
                    $billingModel = $this->generateBilling($receipt, $billing);
                    if ($billingModel == false) {
                        return false;
                    } else {
                        $savedBillings[] = $billingModel;
                    }
                }
            }
        }
        return $savedBillings;
    }


    private function generateBilling($receipt, $billing)
    {
        $billingModel = new Billing();
        $billingModel->receipt_id = $receipt->id;
        $billingModel->payment_method_id = $billing->paymentMethodId;
        $billingModel->billing_charge_id = $billing->billingChargeId;
        $billingModel->customer_id = $this->customerId;
        $billingModel->application_period_id = $this->applicationPeriodId;
        $billingModel->created_by = $this->staffId;
        $billingModel->created_at = date("Y-m-d H:i:s");
        $billingModel->cost = $billing->balance;
        $billingModel->amount_paid = $billing->amountPaid;
        $billingModel->quantity = 1;

        if ($receipt->student_registration_id == true) {
            $billingModel->student_registration_id =
                $receipt->student_registration_id;

            $studentRegistration =
                StudentRegistrationModel::getStudentRegistrationByID(
                    $receipt->student_registration_id
                );
            $billingModel->academic_offering_id =
                $studentRegistration->academicofferingid;
        }

        if ($billingModel->save() == true) {
            return $billingModel;
        } else {
            return null;
        }
    }

    private function billingsSubmitted($billings)
    {
        if ($billings == true) {
            foreach ($billings as $billing) {
                if ($billing->isSelected() == true) {
                    return true;
                }
            }
        }
        return false;
    }

    private function anyBillingsPaidByCheque($billings)
    {
        if (!empty($billings)) {
            $chequePaymentMethod =
                PaymentMethod::find()->where(['name' => 'Cheque'])->one();
            foreach ($billings as $billing) {
                if (
                    $billing->validateModel() == true
                    && $billing->paymentMethodId == $chequePaymentMethod->paymentmethodid
                ) {
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * Generates receipt and billing for enrollment fee payment process
     *
     * @param array BatchStudentFeePaymentBillingForm $billings
     * @return Receipt|ErrorObject
     * 
     * Test command:
     * Untested
     */
    public function processSuccessfulApplicantPaymentRequest($billings)
    {

        if ($this->billingsSubmitted($billings) == false) {
            return new ErrorObject("No fees were selected for payment.");
        } elseif ($this->payUsingCheque == true && $this->chequeNumber == false) {
            return new ErrorObject("Cheque number missing.");
        } elseif (
            $this->payUsingCheque == true
            && $this->anyBillingsPaidByCheque($billings) == false
        ) {
            return new ErrorObject(
                "You must have at least one billing indicating cheque payment."
            );
        } elseif (
            $this->payUsingCheque == false
            && $this->anyBillingsPaidByCheque($billings) == true
        ) {
            return new ErrorObject(
                "You indicated none cheque payment but subsequently added billing with cheque."
            );
        } else {
            $receipt = $this->generateSuccessfulApplicantReceipt($billings);

            if ($receipt == null) {
                return new ErrorObject("Error ocurred generating receipt");
            } else {
                $billings = $this->generateBillings($receipt, $billings);

                if ($billings == false) {
                    return new ErrorObject("Error ocurred generating billings.");
                } else {
                    // $billings = $receipt->getBillings()->all();
                    $customer = UserModel::getUserById($receipt->customer_id);
                    $applicantName = UserModel::getUserFullname($customer);
                    $applicantId = $customer->username;
                    return $receipt;
                }
            }
        }
    }


    public function generateDefaultBillingFormsForStudent(
        $studentRegistration
    ) {
        $records = array();
        $application =
            Application::find()
            ->innerJoin(
                'offer',
                '`application`.`applicationid` = `offer`.`applicationid`'
            )
            ->innerJoin(
                'student_registration',
                '`offer`.`offerid` = `student_registration`.`offerid`'
            )
            ->where([
                // "application.isactive" => 1,
                "application.isdeleted" => 0,
                "student_registration.studentregistrationid" => $studentRegistration->studentregistrationid
            ])
            ->one();

        $applicableBillingCharges =
            BillingChargeModel::getAllBillingChargesForApplication($application);

        foreach ($applicableBillingCharges as $billingCharge) {
            if (StudentModel::isPaymentAllowed(
                $billingCharge,
                $studentRegistration
            ) == true) {
                $record = new BatchStudentFeePaymentBillingForm();
                $record->fillModel($this->customerId, $billingCharge);
                $records[] = $record;
            }
        }

        return $records;
    }


    public function processEnrolledStudentPaymentRequest(
        $billings,
        $studentRegistrationId
    ) {
        if ($this->billingsSubmitted($billings) == false) {
            return new ErrorObject("No fees were selected for payment.");
        } else {
            $receipt = $this->generateEnrolledStudentReceipt(
                // $studentRegistrationId
                $billings
            );

            if ($receipt == null) {
                return new ErrorObject("Error ocurred generating receipt");
            } else {
                $billings = $this->generateBillings($receipt, $billings);

                if ($billings == false) {
                    return new ErrorObject("Error ocurred generating billings.");
                } else {
                    return $receipt;
                }
            }
        }
    }

    private function getPaymentMethodId($billings)
    {
        $uniquePaymentMethodIds = array();
        foreach ($billings as $billing) {
            if (in_array(
                $billing->paymentMethodId,
                $uniquePaymentMethodIds
            ) == false) {
                array_push($uniquePaymentMethodIds, $billing->paymentMethodId);
            }
        }
        if (count($uniquePaymentMethodIds) == 1) {
            return $billings[0]->paymentMethodId;
        } elseif (count($uniquePaymentMethodIds) > 1) {
            $paymentMethod =
                PaymentMethod::find()->where(['name' => 'Combined'])->one();
            if ($paymentMethod == true) {
                return  $paymentMethod->paymentmethodid;
            }
        }
    }


    private function generateEnrolledStudentReceipt($billings)
    {
        $receipt = new Receipt();
        $receipt->payment_method_id = $this->getPaymentMethodId($billings);
        $receipt->customer_id = $this->customerId;

        $studentRegistration =
            $this->getCustomerStudentRegistration(
                $this->customerId,
                $this->applicationPeriodId
            );
        if ($studentRegistration == true) {
            $receipt->student_registration_id =
                $studentRegistration->studentregistrationid;
        }

        $receipt->created_by = $this->staffId;
        $receipt->username = $this->username;
        $receipt->full_name = $this->fullName;
        $receipt->receipt_number = $this->generateReceiptNumber();

        $customer = UserModel::getUserById($this->customerId);
        $receipt->email = $customer->email;

        $receipt->date_paid = $this->datePaid;
        $receipt->timestamp = date("Y-m-d H:i:s");
        if ($receipt->save() == true) {
            return $receipt;
        } else {
            return null;
        }
    }

    // private function generateEnrolledStudentReceipt(
    //     // $studentRegistrationId
    //     $billings
    // ) {
    //     $receipt = new Receipt();
    //     // $receipt->payment_method_id = $this->paymentMethodId; // has to change to method on billing

    //     if ($this->payUsingCheque == true) {
    //         $paymentMethod =
    //             PaymentMethod::find()->where(['name' => 'Cheque'])->one();
    //         if ($paymentMethod == true) {
    //             $receipt->payment_method_id =  $paymentMethod->paymentmethodid;
    //         }
    //     } elseif ($this->payUsingCheque == false) {
    //         $uniquePaymentMethodIds = array();
    //         foreach ($billings as $billing) {
    //             if (in_array(
    //                 $billing->paymentMethodId,
    //                 $uniquePaymentMethodIds
    //             ) == false) {
    //                 array_push($uniquePaymentMethodIds, $billing->paymentMethodId);
    //             }
    //         }
    //         if (count($uniquePaymentMethodIds) > 1) {
    //             $paymentMethod =
    //                 PaymentMethod::find()->where(['name' => 'Combined'])->one();
    //             if ($paymentMethod == true) {
    //                 $receipt->payment_method_id =  $paymentMethod->paymentmethodid;
    //             }
    //         } else {
    //             $receipt->payment_method_id = $billings[0]->paymentMethodId;
    //         }
    //     }


    //     $receipt->customer_id = $this->customerId;

    //     // $receipt->student_registration_id = $studentRegistrationId;

    //     $studentRegistration =
    //         $this->getCustomerStudentRegistration(
    //             $this->customerId,
    //             $this->applicationPeriodId
    //         );
    //     if ($studentRegistration == true) {
    //         $receipt->student_registration_id =
    //             $studentRegistration->studentregistrationid;
    //     }


    //     $receipt->created_by = $this->staffId;
    //     $receipt->username = $this->username;
    //     $receipt->full_name = $this->fullName;
    //     $receipt->receipt_number = $this->generateReceiptNumber();

    //     $customer = UserModel::getUserById($this->customerId);
    //     $receipt->email = $customer->email;

    //     $receipt->date_paid = $this->datePaid;
    //     $receipt->timestamp = date("Y-m-d H:i:s");
    //     if ($receipt->save() == true) {
    //         return $receipt;
    //     } else {
    //         return null;
    //     }
    // }


    // public function generateBillingFormsForPaymentModification(
    //     $receipt,
    //     $maxNewBillingCount
    // ) {
    //     $records = array();

    //     $applicableBillingCharges =
    //         BillingChargeModel::getOutstandingEnrollmentChargesByApplication(
    //             $receipt
    //         );

    //     if (!empty($applicableBillingCharges)) {
    //         foreach ($applicableBillingCharges as $billingCharge) {
    //             $record = new BatchStudentFeePaymentBillingForm();
    //             $record->fillModel($this->customerId, $billingCharge);
    //             $records[] = $record;
    //         }
    //     }

    //     for ($i = 0; $i < $maxNewBillingCount; $i++) {
    //     }

    //     return $records;
    // }

    public function generateBillingFormsForPaymentModification(
        $billings,
        $maxNewBillingCount
    ) {
        $records = array();
        if (!empty($billings)) {
            foreach ($billings as $billing) {
                $billingCharge = $billing->getBillingCharge()->one();
                $billingChargeName = $billingCharge->getBillingType()->one();
                $model = new BatchStudentFeePaymentBillingForm();
                $model->billingChargeId = $billingCharge->id;
                $model->fee = $billingChargeName->name;
                $model->balance = $billing->cost;
                $model->amountPaid = $billing->amount_paid;
                $model->isActive = 1;
                $records[] = $model;
            }
        }

        for ($i = 0; $i < $maxNewBillingCount; $i++) {
            continue;
        }

        return $records;
    }


    private function generateSuccessfulApplicantReceipt($billings)
    {
        $receipt = new Receipt();

        if ($this->payUsingCheque == true) {
            $paymentMethod =
                PaymentMethod::find()->where(['name' => 'Cheque'])->one();
            if ($paymentMethod == true) {
                $receipt->payment_method_id =  $paymentMethod->paymentmethodid;
            }
        } elseif ($this->payUsingCheque == false) {
            $uniquePaymentMethodIds = array();
            foreach ($billings as $billing) {
                if (in_array(
                    $billing->paymentMethodId,
                    $uniquePaymentMethodIds
                ) == false) {
                    array_push($uniquePaymentMethodIds, $billing->paymentMethodId);
                }
            }
            if (count($uniquePaymentMethodIds) > 1) {
                $paymentMethod =
                    PaymentMethod::find()->where(['name' => 'Combined'])->one();
                if ($paymentMethod == true) {
                    $receipt->payment_method_id =  $paymentMethod->paymentmethodid;
                }
            } else {
                $receipt->payment_method_id = $billings[0]->paymentMethodId;
            }
        }

        $receipt->customer_id = $this->customerId;

        $studentRegistration =
            $this->getCustomerStudentRegistration(
                $this->customerId,
                $this->applicationPeriodId
            );
        if ($studentRegistration == true) {
            $receipt->student_registration_id =
                $studentRegistration->studentregistrationid;
        }
        $receipt->created_by = $this->staffId;
        $receipt->username = $this->username;
        $receipt->full_name = $this->fullName;
        $receipt->receipt_number = $this->generateReceiptNumber();

        $customer = UserModel::getUserById($this->customerId);
        $receipt->email = $customer->email;

        $receipt->date_paid = $this->datePaid;
        $receipt->timestamp = date("Y-m-d H:i:s");
        $receipt->cheque_number = $this->chequeNumber;
        if ($receipt->save() == true) {
            return $receipt;
        } else {
            return null;
        }
    }
}
