<?php

namespace common\models;

use Yii;

class ApplicationModel
{
    public static function getApplicationByApplicationID($id)
    {
        return Application::find()->where(["applicationid" => $id])->one();
    }


    public static function getActiveApplicationsByPersonID($id)
    {
        return Application::find()
            ->where(
                [
                    "applicationstatusid" => [2, 3, 4, 5, 6, 7, 8, 9, 10, 11],
                    // "ordering" => [1, 2, 3],
                    "personid" => $id,
                    "isactive" => 1,
                    "isdeleted" => 0
                ]
            )
            ->orderBy("ordering ASC")
            ->all();
    }


    public static function hasActiveApplicationsRelatedToVisibleApplicationPeriod(
        $personid
    ) {
        $idsForVisibleApplicationPeriods =
            ApplicationPeriodModel::getApplicationPeriodIdsForVisibleApplicationPeriods();

        $activeApplications = self::getActiveApplicationsByPersonID($personid);
        foreach ($activeApplications as $application) {
            $offering =
                AcademicOfferingModel::getAcademicOfferingByID(
                    $application->academicofferingid
                );

            if (in_array($offering->applicationperiodid, $idsForVisibleApplicationPeriods) == true) {
                return true;
            }
        }
        return false;
    }


    public static function getFormattedProgrammeChoice($application)
    {
        $capeSubjectsNames = array();

        $offering =
            AcademicOfferingModel::getAcademicOfferingByID(
                $application->academicofferingid
            );

        $programme =
            ProgrammeCatalogModel::getProgrammeCatalogByID(
                $offering->programmecatalogid
            );

        $applicationCapeSubjects =
            ApplicationCapesubjectModel::getApplicationCapeSubjectsByApplicationID(
                $application->applicationid
            );

        foreach ($applicationCapeSubjects as $cs) {
            $capeSubjectsNames[] =
                CapeSubjectModel::getCapeSubjectNameByID($cs->capesubjectid);
        }

        if (empty($applicationCapeSubjects) == true) {
            // return ProgrammeCatalogModel::getFormattedProgrammeName($programme);
            return AcademicOfferingModel::getFormattedOfferingName($offering);
        } else {
            $academicYear =
                AcademicYearModel::getAcademicYearByID($offering->academicyearid);

            return $programme->name
                . " ({$academicYear->title}): "
                . implode(' ,', $capeSubjectsNames);
        }
    }


    public static function formatApplicationsInformation($applications)
    {
        $data = array();
        foreach ($applications as $application) {
            $row = array();
            if ($application->ordering === 1) {
                $row["ordering"] = "First Choice";
            } elseif ($application->ordering === 2) {
                $row["ordering"] = "Second Choice";
            } elseif ($application->ordering === 3) {
                $row["ordering"] = "Third Choice";
            }

            $row["name"] = self::getFormattedProgrammeChoice($application);
            $data[] = $row;
        }
        return $data;
    }


    public static function getApplicationPeriodName($application)
    {
        $academicOffering =
            AcademicOfferingModel::getAcademicOfferingByID(
                $application->academicofferingid
            );

        if ($academicOffering == true) {
            $period =
                ApplicationPeriodModel::getApplicationPeriodByID(
                    $academicOffering->applicationperiodid
                );

            return $period->name;
        }
        return null;
    }

    public static function getApplicationPeriod($application)
    {
        $academicOffering =
            AcademicOfferingModel::getAcademicOfferingByID(
                $application->academicofferingid
            );

        if ($academicOffering == true) {
            return ApplicationPeriodModel::getApplicationPeriodByID(
                $academicOffering->applicationperiodid
            );
        }
        return null;
    }


    public static function getSuccessfulApplication($applicant)
    {
        return Application::find()
            ->where([
                "personid" => $applicant->personid,
                "applicationstatusid" => 9,
                "isactive" => 1,
                "isdeleted" => 0
            ])
            ->one();
    }


    public static function getFormattedProgrammeChoices($personId)
    {
        $applications = self::getActiveApplicationsByPersonID($personId);
        return self::formatApplicationsInformation($applications);
    }


    public static function getSubjectsForCapeApplication($application)
    {
        return CapeSubject::find()
            ->innerJoin(
                'application_capesubject',
                '`cape_subject`.`capesubjectid` = `application_capesubject`.`capesubjectid`'
            )
            ->innerJoin(
                'application',
                '`application_capesubject`.`applicationid` = `application`.`applicationid`'
            )
            ->where([
                "cape_subject.isactive" => 1,
                "cape_subject.isdeleted" => 0,
                "application.personid" => $application->personid,
                "application.academicofferingid" => $application->academicofferingid,
                "application.applicationstatusid" => 9
            ])
            ->all();
    }


    public static function isCape($application)
    {
        $academicOffering =
            AcademicOfferingModel::getAcademicOfferingByID(
                $application->academicofferingid
            );

        $programmeCatalog =
            ProgrammeCatalogModel::getProgrammeCatalogByID(
                $academicOffering->programmecatalogid
            );

        if ($programmeCatalog->name === "CAPE") {
            return true;
        }

        return false;
    }


    public static function getProgrammeName($application)
    {
        $programmeDetails = null;
        $capeSubjectsNames = array();

        $offering =
            AcademicOffering::find()
            ->where(["academicofferingid" => $application->academicofferingid])
            ->one();

        $programme =
            ProgrammeCatalog::find()
            ->where(["programmecatalogid" => $offering->programmecatalogid])
            ->one();

        $capeSubjects =
            ApplicationCapesubject::find()
            ->where([
                "applicationid" => $application->applicationid,
                "isactive" => 1,
                "isdeleted" => 0
            ])
            ->all();

        foreach ($capeSubjects as $cs) {
            $capeSubjectsNames[] =
                CapeSubject::find()->where(['capesubjectid' => $cs->capesubjectid])->one()->subjectname;
        }


        $programmeSpecialisations = "";
        $specialisationsDescriptionCollection = array();
        $applicationSpecialisations =  ApplicationSpecialisation::find()
            ->where([
                "application_id" => $application->applicationid,
                "is_deleted" => 0
            ])
            ->all();
        if ($applicationSpecialisations == true) {
            foreach ($applicationSpecialisations as $applicationSpecialisation) {
                $valueAsString = "";
                $academicOfferingSpecialisation =
                    AcademicOfferingSpecialisation::find()
                    ->where([
                        "id" => $applicationSpecialisation->academic_offering_specialisation_id,
                        "is_deleted" => 0
                    ])
                    ->one();

                $programmeSpecialisation =
                    ProgrammeSpecialisation::find()
                    ->where([
                        "id" => $academicOfferingSpecialisation->programme_specialisation_id,
                        "is_deleted" => 0
                    ])
                    ->one();

                $valueAsString .= $programmeSpecialisation->name;

                $programmeSpecialisationType =
                    ProgrammeSpecialisationType::find()
                    ->where([
                        "id" => $academicOfferingSpecialisation->programme_specialisation_type_id,
                        "is_deleted" => 0
                    ])
                    ->one();

                if ($programmeSpecialisationType->name == "General Specialisation") {
                    $valueAsString = "Specialisation - {$programmeSpecialisation->name}";
                } else {
                    $valueAsString =
                        "{$programmeSpecialisationType->name} - {$programmeSpecialisation->name}";
                }

                array_push(
                    $specialisationsDescriptionCollection,
                    $valueAsString
                );
            }
            $programmeSpecialisations = implode(" <br/> ", $specialisationsDescriptionCollection);
        }


        if (empty($capeSubjects) == true) {
            $qualificationTypeAbbreviation =
                QualificationType::find()
                ->where(["qualificationtypeid" => $programme->qualificationtypeid])
                ->one()->abbreviation;

            $name = $programme->name;
            $specialisation = $programme->specialisation;

            if (
                $specialisation != false
                && $specialisation != null
                && $specialisation != ""
                && $specialisation != " "
            ) {
                $programmeDetails =
                    "{$qualificationTypeAbbreviation}. {$name} ({$specialisation})";
            } else {
                $programmeDetails =
                    "{$qualificationTypeAbbreviation}. {$name}";
            }

            if ($programmeSpecialisations == true) {
                $programmeDetails .= ",<br/> {$programmeSpecialisations}";
            }
        } else {
            $programmeDetails =
                "{$programme->name} : " . implode(' ,', $capeSubjectsNames);
        }
        return $programmeDetails;
    }
}
